# coding: utf-8
import copy
import json
import logging
import requests
import gzip
import os

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.types import client as ctc
import sandbox.common.types.resource as ctr


CONFIG = {
    "test": {
        "MODERATED_FEED_URL": "https://yadi-test.sec.yandex-team.ru/api/v1/moderate/list/all?gt={}",
        "ACTUAL_FEED": "https://yadi-test.sec.yandex-team.ru/db/web.json",
        "VULN_URL_PREFIX": "https://yadi-test.sec.yandex-team.ru/vulns/vuln/{}",
        "QLOUD_ENVIRONMENT": ["security.yadi.test", "security.yadi.yaudit-dev"],
    },
    "prod": {
        "MODERATED_FEED_URL": "https://yadi.yandex-team.ru/api/v1/moderate/list/all?gt={}",
        "ACTUAL_FEED": "https://yadi.yandex-team.ru/db/web.json",
        "VULN_URL_PREFIX": "https://yadi.yandex-team.ru/vulns/vuln/{}",
        "QLOUD_ENVIRONMENT": ["security.yadi.prod", "security.yadi.yaudit"],
    }
}

# w/o this fields task the task throw exception
REQUIRED_FIELDS = [
    "id",
    "title",
    "module_name",
    "cvss_score",
    "vulnerable_versions",
    "language",
    "patched_versions",
    "patch_exists",
    "reference",
    "disclosed",
    "rich_description"
]

# fields for [language].json
YADI_FIELDS = [
    "id",
    "title",
    "module_name",
    "cvss_score",
    "vulnerable_versions",
    "language",
    "patched_versions",
    "patch_exists",
    "reference"
]

YADI_SCHEMA = {
    "type": "array",
    "items": {
        "type": "object",
        "additionalProperties": True,
        "properties": {
            "id": {
                "type": "string"
            },
            "title": {
                "type": "string"
            },
            "module_name": {
                "type": "string"
            },
            "language": {
                "type": "string"
            },
            "cvss_score": {
                "type": "number"
            },
            "vulnerable_versions": {
                "type": "string"
            },
            "patched_versions": {
                "type": ["string", "null"]
            },
            "patch_exists": {
                "type": "boolean"
            },
            "reference": {
                "type": "string",
                "format": "url"
            },
            "description": {
                "type": ["string", "null"]
            },
            "disclosed": {
                "type": "number"
            },
            "banned": {
                "type": "boolean"
            },
            "external_references": {
                "type": "array",
                "items": {
                    "type": "object",
                    "additionalProperties": True,
                    "properties": {
                        "title": {
                            "type": ["string", "null"]
                        },
                        "url": {
                            "type": "string",
                            "format": "url"
                        }
                    },
                    "required": ["url"]
                }
            },
            "raw_id": {
                "type": "string"
            },
            "rich_description": {
                "type": "boolean"
            }
        },
        "required": REQUIRED_FIELDS
    }
}


class YadiDatabaseResourse(sdk2.Resource):
    """ Yadi DB (.json) resourse """
    releasers = ["robot-yadi"]
    release_subscribers = ["robot-yadi"]
    releasable = True
    share = True
    restart_policy = ctr.RestartPolicy.DELETE


class YadiFeedDeploy(sdk2.Task):
    config = {}

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 1024
        ram = 2048
        environments = (
            environments.PipEnvironment('pyrsistent', version='0.16.0'),
            environments.PipEnvironment("jsonschema", "3.0.2", custom_parameters=['--upgrade-strategy', 'only-if-needed']),
        )

    class Parameters(sdk2.Task.Parameters):
        deploy = sdk2.parameters.Bool("Deploy feed to Qloud (otherwise just create sandbox resources)", default=True)

        with sdk2.parameters.RadioGroup("Env type") as env_type:
            env_type.values["prod"] = env_type.Value(value="prod", default=True)
            env_type.values["test"] = env_type.Value(value="test")

    def on_execute(self):
        self.config = CONFIG.get(self.Parameters.env_type)

        vulns = self.fetch_moderated_feed()
        self.validate_vulns(vulns)
        filtered_vulns = self.filter_vulns(vulns, REQUIRED_FIELDS)
        sorted_vulns = self.sort_vulns(vulns)

        result_dir = sdk2.Path("yadi")
        yadi_db_resourse = YadiDatabaseResourse(self, "Yadi vulnerabilities database", result_dir)
        yadi_db_data = sdk2.ResourceData(yadi_db_resourse)
        yadi_db_data.path.mkdir(0o755, parents=True, exist_ok=True)

        self.create_yadi_web(yadi_db_data.path, sorted_vulns)
        self.create_yadi_db(yadi_db_data.path, filtered_vulns)

        files = os.listdir(str(yadi_db_data.path))
        index_data = json.dumps({
            "files": files
        })
        yadi_db_data.path.joinpath("index.json").write_bytes(index_data)

        logging.info("Create resource")
        yadi_db_data.ready()

        if self.Parameters.deploy:
            actual_vulns = self.fetch_actual_feed()
            if self.yadi_web_changed(actual_vulns, sorted_vulns):
                self.deploy_service(yadi_db_resourse.id)
            else:
                logging.info("deployment canceled because the web.json has not changed")

    def fetch_moderated_feed(self):
        vulns = []
        is_last = False
        url = self.config.get("MODERATED_FEED_URL").format("0")
        session = requests.Session()
        session.headers["Content-Type"] = "application/json"
        while not is_last:
            logging.info("requested feed url: %s" % url)
            resp = session.get(url, verify=True)

            if resp.status_code != 200:
                raise Exception("Failed to fetch feed (%d): %s" % (resp.status_code, resp.content))

            result = resp.json()
            if not result.get("ok"):
                raise Exception("Failed to fetch feed: %s" % result.get("error"))

            url = result.get("result").get("next_page")
            is_last = result.get("result").get("is_last")

            for vuln in result.get("result").get("vulns"):
                vulns.append(self.cook_vuln(vuln))

        return vulns

    def fetch_actual_feed(self):
        url = self.config.get("ACTUAL_FEED")
        session = requests.Session()
        session.headers["Content-Type"] = "application/json"
        resp = session.get(url, verify=True)

        if resp.status_code != 200:
            raise Exception("Failed to fetch feed (%d): %s" % (resp.status_code, resp.content))

        return resp.json()

    def cook_vuln(self, feed_vuln):
        """ Modify yadi-web feed vulnerability to yadi vulnerability """
        feed_vuln["raw_id"] = feed_vuln.get("id")
        feed_vuln["id"] = feed_vuln.pop("yadi_id")
        feed_vuln["module_name"] = feed_vuln.pop("package")
        feed_vuln["severity"] = cvs_to_severity(feed_vuln.get("cvss_score"))
        feed_vuln["patched_versions"] = feed_vuln.get("patched_versions")
        feed_vuln["external_references"] = feed_vuln.pop("references", None)
        feed_vuln["reference"] = self.config.get("VULN_URL_PREFIX").format(feed_vuln.get("id"))
        return feed_vuln

    def validate_vulns(self, vulns):
        from jsonschema import validate
        logging.info("Starts vulns validating")
        validate(vulns, YADI_SCHEMA)

    def vulns_by_lang(self, vulns):
        result = {}
        for vuln in vulns:
            lang = vuln.get("language")
            if lang not in result.keys():
                result[lang] = []
            result[lang].append(vuln)
        return result

    def filter_vulns(self, vulns, keys):
        filtered_vulns = copy.deepcopy(vulns)
        for vuln in filtered_vulns:
            for key in vuln.keys():
                # Remove unused fields
                if key not in keys:
                    vuln.pop(key, None)

            # checking important fields
            if set(vuln.keys()) != set(keys):
                raise Exception("Important vuln ({}) fields doesn't exist: {}".format(
                    vuln.get("id"),
                    ", ".join(str(x) for x in list(set(keys) - set(vuln.keys())))
                    ))
        return filtered_vulns

    def sort_vulns(self, vulns):
        logging.info("Starts vulns sorting")
        return sorted(vulns, key=lambda vuln: vuln.get("disclosed"), reverse=True)  # new vulns in the top

    def create_yadi_db(self, resourse_path, vulns):
        logging.info("Starts yadi db creation in: %s", resourse_path)

        filtered_vulns = self.filter_vulns(vulns, YADI_FIELDS)
        data = self.vulns_by_lang(filtered_vulns)
        for lang, lang_vulns in data.items():
            resourse_path.joinpath("{}.json".format(lang)).write_bytes(json.dumps(lang_vulns))
            write_gzip(resourse_path.joinpath("{}.json.gz".format(lang)), json.dumps(lang_vulns))

    def create_yadi_web(self, resourse_path, vulns):
        logging.info("Starts web db creation in: %s", resourse_path)
        resourse_path.joinpath("web.json").write_bytes(json.dumps(vulns))

    def yadi_web_changed(self, old_vulns, new_vulns):
        old_vulns_dict, new_vulns_dict = {}, {}
        for vuln in old_vulns:
            old_vulns_dict[vuln.get("id")] = vuln

        for vuln in new_vulns:
            new_vulns_dict[vuln.get("id")] = vuln

        return old_vulns_dict != new_vulns_dict

    def deploy_service(self, db_id):
        for environment in self.config.get("QLOUD_ENVIRONMENT"):
            session = requests.Session()
            session.headers["Authorization"] = "OAuth {}".format(sdk2.Vault.data("buglloc", "YADI_TOKEN"))
            session.headers["Content-Type"] = "application/json"

            logging.info("Fetch env: %s", environment)
            resp = session.get("https://platform.yandex-team.ru/api/v1/environment/dump/%s" % environment)
            if resp.status_code != 200:
                raise Exception("Failed to fetch env (%d): %s" % (resp.status_code, resp.content))

            state = resp.json()
            for component in state["components"]:
                for res in component["sandboxResources"]:
                    if res["localName"] != "yadi":
                        continue
                    res["id"] = db_id

            logging.info("Upload env: %s", environment)
            resp = session.post("https://platform.yandex-team.ru/api/v1/environment/upload", json=state)
            if resp.status_code != 200:
                raise Exception("Failed to upload env (%d): %s" % (resp.status_code, resp.content))


def cvs_to_severity(score):
    # See CVSSv3 for details: https://www.first.org/cvss/specification-document#i5
    if score < 3.9:
        return "low"
    elif score < 6.9:
        return "medium"
    elif score < 8.9:
        return "high"
    return "critical"


def write_gzip(path, content):
    with gzip.open(str(path), "wb") as f:
        f.write(content)
