# coding: utf-8
import os
import tempfile

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.common.types import client as ctc
from sandbox.common.types.misc import DnsType
from sandbox.projects.security.common.task import SecurityBaseTask


class FeedSnatch(SecurityBaseTask):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        container_resource = 1598242028
        disk_space = 4096
        ram = 4096
        cores = 8
        dns = DnsType.DNS64

    class SecurityOptions(SecurityBaseTask.SecurityOptions):
        subtask_resource_attrs = {
            "target": "security/yadi/snatcher/cmd/snatcher"
        }

    class Parameters(sdk2.Task.Parameters):
        verbose_mode = sdk2.parameters.Bool("Verbose mode", default=False)
        id_mapping = sdk2.parameters.Bool("Old vuln IDs mappin", default=True)
        email = sdk2.parameters.Bool("Send email notification with new vulns", default=True)
        debug = sdk2.parameters.Bool("Debug SB task (suspend after execution)", default=False)

        with sdk2.parameters.RadioGroup("Env type") as env_type:
            env_type.values["prod"] = env_type.Value(value="prod", default=True)
            env_type.values["dev"] = env_type.Value(value="dev")
        with sdk2.parameters.RadioGroup("Command") as command:
            command.values["list"] = command.Value(value="ls")
            command.values["update"] = command.Value(value="update", default=True)
        with sdk2.parameters.RadioGroup("Target") as target:
            target.values["snyk"] = target.Value(value="snyk")
            target.values["wiki"] = target.Value(value="wiki")
            target.values["linux-ubuntu"] = target.Value(value="linux-ubuntu")
            target.values["linux-debian"] = target.Value(value="linux-debian")
            target.values["linux-alpine"] = target.Value(value="linux-alpine")
            target.values["linux-kernel"] = target.Value(value="linux-kernel")
            target.values["all"] = target.Value(value="all", default=True)

    def get_cmd(self, binary_path, target):
        cmd = [
            binary_path,
            self.Parameters.command,
            target,
            "--env", self.Parameters.env_type,
        ]

        if self.Parameters.verbose_mode:
            cmd.append("--verbose")

        if self.Parameters.id_mapping and self.Parameters.command == "update":
            cmd.append("--map")

        if self.Parameters.email and self.Parameters.command == "update":
            cmd.append("--email")

        return cmd

    def get_additional_env(self):
        return {
            "SNYK_CONSUMER_ID": sdk2.Vault.data("YADI_SNATCHER_SNYK_CONSUMER_ID"),
            "SNYK_SECRET": sdk2.Vault.data("YADI_SNATCHER_SNYK_SECRET"),
            "YADI_SNATCHER_OAUTH_TOKEN": sdk2.Vault.data("YADI_SNATCHER_OAUTH_TOKEN"),
        }

    def on_execute(self):
        if self.Parameters.debug:
            self.suspend()

        # move to snatcher?
        go_path = tempfile.NamedTemporaryFile().name
        os.mkdir(go_path)

        targets = []
        if self.Parameters.target == "all":
            targets = ["snyk", "linux-ubuntu", "linux-debian", "linux-alpine", "linux-kernel", "wiki"]
        else:
            targets = [self.Parameters.target]

        for target in targets:
            with sdk2.helpers.ProcessLog(self, logger="security_subtask_"+target) as pl:
                cmd = self.get_cmd(self.subtask_binary_path, target)
                env = self.get_env()
                env["GOPATH"] = go_path
                subprocess.check_call(cmd, shell=False, stdout=pl.stdout, stderr=pl.stderr, env=env)
