# coding: utf-8
import os
import shutil
import json
import logging
import requests
import time
import tempfile
from jinja2 import Template

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects import resource_types
from yasyslog import YaSyslog


"""
Example:
  * **!!(red)send@0.3.0: Elevation Of Privileges!!**:
  Используется в цепочке //express@4.1.2// > //serve-static@1.1.0// > //send@0.3.0//
  Починено в версиях **>=0.8.4**, необходимо обновить **express@4.1.2** до **express@4.8.0** (повлечет обновление //serve-static@1.5.4// > //send@0.8.5//)
  Подробнее в https://yadi.yandex-team.ru/vulns/vuln/sc:1316:51024355
"""
TEMPLATE = u'''{%- set col = cvss_color(cvss_score) -%}
  * **{% if col %}!!({{col}}){% endif %}{{package_name}}@{{version}}: {{summary}}{% if col %}!!{% endif %}**:
  {% if path|length > 1 %}Используется в цепочке //{{path|join('// > //')}}// {% endif %}
  Починено в версиях **{{patched_versions}}**{% if suggest %}, {{ format_suggest(suggest, path) }}{% endif %}
  Подробнее в {{reference}}
'''
YADI_HOST = 'https://yadi.yandex-team.ru'


class YadiMonitor(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        send_to_splunk = sdk2.parameters.Bool('Send results to Splunk', default=True)
        yadi_binary = sdk2.parameters.Resource(
            'Yadi executable',
            resource_type=resource_types.YADI_BINARY,
            required=True
        )

    tmpl = None

    def on_execute(self):
        monitor_id = int(time.time())

        yadi_path = str(sdk2.ResourceData(self.Parameters.yadi_binary).path)
        logging.info('Yadi tool saved to: %s' % yadi_path)

        tmp_dir = tempfile.mkdtemp()
        package_info_path = os.path.join(tmp_dir, 'yadi.json')
        session = requests.Session()
        session.headers['x-token'] = sdk2.Vault.data('YADI_INTERNAL_TOKEN')
        session.headers['Content-Type'] = 'application/json'

        next_url = YADI_HOST + '/api/v1/internal/projects'
        while next_url:
            logging.info('Retrieve projects from: %s' % next_url)
            resp = session.get(
                next_url,
                verify=False  # Skynet, I look at you
            )

            result = resp.json()
            if not result['ok']:
                logging.error('Failed to retrieve projects: %s' % resp.text)
                break
            logging.info('Retrieved %d projects' % len(result['result']['projects']))

            next_url = result['result']['next']
            for project in result['result']['projects']:
                project_name = '%s/%s' % (project['service'], project['name'])
                logging.info('Process "%s" project' % project_name)
                resp = session.get(
                    YADI_HOST + '/api/v1/project/%s/dependencies' % project_name,
                    verify=False  # Skynet, I look at you
                )

                result = resp.json()
                if not result['ok']:
                    logging.error('Failed to retrieve project "%s" deps: %s' % (project_name, resp.text))
                    continue

                if not result['result']:
                    logging.info('No deps')
                    continue

                severity = self.cvs_to_cli(project.get('severity', 0.0))
                yadi_args = [yadi_path, 'test', severity, '--exit-status', '0', '--format', 'json', package_info_path]
                for dep in result['result']:
                    with open(package_info_path, 'w') as temp:
                        temp.write(json.dumps(dep))
                        temp.flush()

                    yadi_result = None
                    # with sdk2.helpers.ProcessLog(self, "yadi") as process_log:
                    p = sp.Popen(
                        yadi_args,
                        stdout=sp.PIPE, stderr=sp.PIPE
                    )
                    stdout, stderr = p.communicate()
                    p.wait()

                    logging.info('Stdout: %s' % stdout)
                    logging.info('Stderr: %s' % stderr)

                    if 'No issues found.' in stdout:
                        continue

                    if stdout:
                        yadi_result = json.loads(stdout)

                    if not yadi_result:
                        logging.error('No results for project "%s"' % project_name)
                        continue

                    if not self.Parameters.send_to_splunk:
                        continue

                    with YaSyslog('Yadi') as log:
                        for res in yadi_result:
                            if not res.get('issues', None):
                                # No issues found
                                continue

                            for issue in res['issues']:
                                entry = {
                                    'scan_id': monitor_id,
                                    'project': project_name,
                                    'user': project['owners'][0],
                                    'followers': ' '.join(project['owners'][1:]),
                                    'path_dep': dep.get('path', 'unknown'),
                                    'issue_id': issue['id'],
                                    'cvss': issue['cvss_score'],
                                    'severity': self.cvs_to_severity(issue['cvss_score']),
                                    'summary': issue['summary'],
                                    'xref': issue['reference'],
                                    'patched_versions': issue['patched_versions'],
                                    'from_dep': ' '.join(issue['path']),
                                    'to_dep': ' '.join(issue['suggest']),
                                    'ticket_message': self.format_issue(issue)
                                }
                                log_entry = u''
                                for name, val in entry.items():
                                    log_entry += u'%s="%s" ' % (name, val)

                                log.warn(log_entry)
        shutil.rmtree(tmp_dir)

    def cvs_to_cli(self, score):
        # See CVSSv3 for details: https://www.first.org/cvss/specification-document#i5
        if score < 0.1:
            return '-ll'
        elif score < 3.9:
            return '-l'
        elif score < 6.9:
            return '-ll'
        elif score < 8.9:
            return '-lll'
        else:
            return '-llll'

    def cvs_to_severity(self, score):
        # See CVSSv3 for details: https://www.first.org/cvss/specification-document#i5
        if score < 0.1:
            return 'info'
        elif score < 3.9:
            return 'low'
        elif score < 6.9:
            return 'medium'
        elif score < 8.9:
            return 'high'
        else:
            return 'critical'

    def format_issue(self, issue):
        if not self.tmpl:
            self.tmpl = Template(TEMPLATE)
            self.tmpl.globals['cvss_color'] = cvss_color
            self.tmpl.globals['format_suggest'] = format_suggest

        result = self.tmpl.render(**issue)
        return result.replace('"', "'").replace('\n', '\\n').replace('\r', '\\r')


def cvss_color(score):
    if score < 0.1:
        return 'gray'
    elif score < 3.9:
        return 'gray'
    elif score < 6.9:
        return ''
    elif score < 8.9:
        return 'red'
    else:
        return 'red'


def format_suggest(suggest, path):
    if not suggest:
        return ''

    result = ''
    if path[0] == suggest[0]:
        result = u'необходимо переустановить  **{}**'.format(path[0])
    else:
        result = u'необходимо обновить **{}** до **{}**'.format(path[0], suggest[0])

    if len(suggest) > 1:
        result += u' (повлечет обновление //{}//)'.format('// > //'.join(suggest[1:]))

    return result


"""
[
  {
    "path": "/home/buglloc/go/src/github.yandex-team.ru/product-security/yadi/yadi.json",
    "issues": [
      {
        "id": "sc:2475:60511531",
        "packageName": "negotiator",
        "cvss_score": 7.8,
        "summary": "Regular Expression Denial Of Service (ReDoS)",
        "reference": "https://yadi.common.yandex.net/vulns/vuln/sc:2475:60511531",
        "patched_versions": ">=0.6.1",
        "version": "0.4.9",
        "suggest": [
          "express@4.14.0",
          "accepts@1.3.3",
          "negotiator@0.6.1"
        ],
        "path": [
          "express@4.1.2",
          "accepts@1.0.1",
          "negotiator@0.4.9"
        ]
      },
      {
        "id": "sc:1316:51024355",
        "packageName": "send",
        "cvss_score": 7.5,
        "summary": "Elevation Of Privileges",
        "reference": "https://yadi.common.yandex.net/vulns/vuln/sc:1316:51024355",
        "patched_versions": ">=0.8.4",
        "version": "0.3.0",
        "suggest": [
          "express@4.8.0",
          "serve-static@1.5.4",
          "send@0.8.5"
        ],
        "path": [
          "express@4.1.2",
          "serve-static@1.1.0",
          "send@0.3.0"
        ]
      },
      {
        "id": "sc:1316:51024355",
        "packageName": "send",
        "cvss_score": 7.5,
        "summary": "Elevation Of Privileges",
        "reference": "https://yadi.common.yandex.net/vulns/vuln/sc:1316:51024355",
        "patched_versions": ">=0.8.4",
        "version": "0.3.0",
        "suggest": [
          "express@4.8.8",
          "send@0.8.5"
        ],
        "path": [
          "express@4.1.2",
          "send@0.3.0"
        ]
      }
    ]
  }
]
"""
