import socket


class Facility:
    "Syslog facilities"
    (
        KERN, USER, MAIL, DAEMON, AUTH, SYSLOG,
        LPR, NEWS, UUCP, CRON, AUTHPRIV, FTP,
    ) = range(12)

    (
        LOCAL0, LOCAL1, LOCAL2, LOCAL3,
        LOCAL4, LOCAL5, LOCAL6, LOCAL7,
    ) = range(16, 24)


class Level:
    "Syslog levels"
    (
        EMERG, ALERT, CRIT, ERR,
        WARNING, NOTICE, INFO, DEBUG,
    ) = range(8)


class YaSyslog:
    """A syslog client that logs to a remote server.

    Example:
    >>> log = YaSyslog("some", host="foobar.example")
    >>> log.send("hello", Level.WARNING)
    """

    def __init__(self, name, host='trapdoor.yandex.net', port=514, facility=Facility.USER):
        self.host = host
        self.port = port
        self.facility = facility
        self.name = name
        self.socket = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)

    def send(self, message, level):
        "Send a syslog message to remote host using TCP."
        data = u'<{priority}>{progname}: {message}\n'.format(
            priority=(level + self.facility * 8),
            progname=self.name,
            message=message
        )

        self.socket.sendall(data.encode('utf-8'))

    def warn(self, message):
        "Send a syslog warning message."
        self.send(message, Level.WARNING)

    def notice(self, message):
        "Send a syslog notice message."
        self.send(message, Level.NOTICE)

    def error(self, message):
        "Send a syslog error message."
        self.send(message, Level.ERR)

    def __enter__(self):
        self.socket.connect((self.host, self.port))
        return self

    def __exit__(self, exception_type, exception_value, traceback):
        self.socket.close()
