# coding: utf-8

import logging
import json

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.common.types import client as ctc
import sandbox.common.types.resource as ctr
from sandbox.projects.security.common.task import SecurityBaseTask
from sandbox.projects.security.common.helpers import cached_property
from hec_sender import HecSender


logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


class YadiArcResults(sdk2.Resource):
    any_arch = True
    releasers = ["robot-yadi"]
    release_subscribers = ["robot-yadi"]
    releasable = True
    share = True
    restart_policy = ctr.RestartPolicy.DELETE


class YadiArc(SecurityBaseTask):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 4096
        ram = 4096
        cores = 8

    class SecurityOptions(SecurityBaseTask.SecurityOptions):
        subtask_resource_attrs = {
            "target": "security/yadi/yadi-arc/cmd/yadi-arc"
        }

    class Parameters(sdk2.Task.Parameters):
        send_to_splunk = sdk2.parameters.Bool(
            "Send scan results to Splunk", default=True)
        resolve_arcadia_groups = sdk2.parameters.Bool(
            "Resolve owners group", default=True)
        check_exlcudes_file = sdk2.parameters.Bool(
            "check arcadia-wide excludes file", default=True)

        with sdk2.parameters.CheckGroup("Languages") as langs:
            langs.values.python = "python"
            langs.values.golang = "golang"

    @cached_property
    def arcadia_root(self):
        return sdk2.svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia@HEAD")

    @cached_property
    def arcadia_revision(self):
        return sdk2.svn.Arcadia.get_revision(self.arcadia_root)

    @property
    def hec_sender(self):
        return HecSender(
            token=sdk2.Vault.data("YADI_ARC_HEC_TOKEN"),
            index="yandex_events",
            source="yadi-arc",
            source_type="yadi"
        )

    def get_author(self, path):
        return sdk2.svn.Arcadia.info(self.arcadia_root + path).get("author", "")

    def get_cmd(self, binary_path, lang):
        cmd = [
            binary_path,
            "contrib",
            str(lang),
        ]

        if self.Parameters.resolve_arcadia_groups:
            cmd.append("--expand-groups")

        if self.Parameters.check_exlcudes_file:
            cmd.append("--global-excludes")

        cmd.extend([
            "--verbose",
            "--format", "json",
            "--exit-status", "0",
            self.arcadia_root,
        ])

        return cmd

    def analyze(self, lang, out_file_path):
        logging.info("Starting yadi-arc with output to '%s'", out_file_path)
        filename = "yadi-arc_{}.json".format(lang)
        resource_full_path = out_file_path.joinpath(filename)
        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            cmd = self.get_cmd(self.subtask_binary_path, lang)
            env = self.get_env()

            with resource_full_path.open("wb") as fd:
                subprocess.check_call(
                    cmd, shell=False, stdout=fd, stderr=pl.stderr, env=env)
            return json.loads(resource_full_path.read_bytes())
        return None

    def on_execute(self):
        if not self.Parameters.langs:
            raise Exception("no languages configured")

        result_dir = sdk2.Path("yadi_arc")
        yadi_arc_resourse = YadiArcResults(
            self, "YadiArc results", result_dir)
        yadi_arc_data = sdk2.ResourceData(yadi_arc_resourse)
        yadi_arc_data.path.mkdir(0o755, parents=True, exist_ok=True)

        with self.hec_sender as hec:
            for lang in list(self.Parameters.langs):
                results = self.analyze(lang, yadi_arc_data.path)

                if not results:
                    continue

                if not self.Parameters.send_to_splunk:
                    continue

                for vulnerable_contrib in results:
                    contrib = vulnerable_contrib
                    contrib["path"] = trim_prefix(
                        contrib["path"], self.arcadia_root)
                    contrib["url"] = 'https://a.yandex-team.ru/arc/trunk/arcadia{path}?rev={rev}'.format(
                        path=contrib["path"],
                        rev=self.arcadia_revision
                    )
                    last_contributor = self.get_author(vulnerable_contrib["path"])
                    contrib["owners"] = sort_owners(contrib["owners"], last_contributor)
                    for issue in contrib["issues"]:
                        if issue["patched_versions"] == "":
                            issue["patched_versions"] = "-"

                    logger.info("sending contrib to splunk: '%s'",
                                contrib["path"])
                    hec.send(
                        lang=lang,
                        revision=self.arcadia_revision,
                        contrib=contrib
                    )

        yadi_arc_data.ready()


def trim_prefix(text, prefix):
    if text.startswith(prefix):
        return text[len(prefix):]
    return text


def sort_owners(owners, last_contributor):
    if len(owners) > 1 and "pg" == owners[0]:
        owners = owners[1:]

    if last_contributor not in owners:
        return owners

    i = owners.index(last_contributor)
    if i > 0:
        owners.insert(0, owners.pop(i))

    return owners
