# coding: utf-8

import logging

from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.common.types import client as ctc
import sandbox.common.types.resource as ctr

from sandbox.projects.security.common.task import SecurityBaseTask
from sandbox.projects.security.common.helpers import cached_property
from sandbox.projects.devtools.ChangesDetector import ChangesDetector

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


class YadiArcPolicyResult(sdk2.Resource):
    any_arch = True
    releasers = ["robot-yadi"]
    release_subscribers = ["robot-yadi"]
    releasable = True
    share = True
    restart_policy = ctr.RestartPolicy.DELETE


class YadiArcPolicyZipatch(sdk2.Resource):
    any_arch = True
    releasers = ["robot-yadi"]
    release_subscribers = ["robot-yadi"]
    releasable = True
    share = True
    restart_policy = ctr.RestartPolicy.DELETE


class YadiArcPolicy(SecurityBaseTask):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 4096
        ram = 4096
        cores = 8

    class SecurityOptions(SecurityBaseTask.SecurityOptions):
        subtask_resource_attrs = {
            "target": "security/yadi/yadi-arc/cmd/yadi-arc"
        }

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("arcadia"):
            arcadia_url = sdk2.parameters.String(
                label='Arcadia path to be mounted (only SVN API supported, must contain revision)',
                default=None
            )

        with sdk2.parameters.RadioGroup("Strategy") as strategy:
            strategy.values["build"] = strategy.Value("build", default=True)
            strategy.values["resource"] = strategy.Value("resource")

        with sdk2.parameters.Group("resource"):
            patch = sdk2.parameters.Resource(
                label='Zipatch with which Arcadia should be built',
                required=False,
                default=None)

        with sdk2.parameters.Group("build"):
            with sdk2.parameters.RadioGroup("Languages") as lang:
                lang.values["java"] = lang.Value("java", default=True)

        with sdk2.parameters.Group("subtask"):
            apply_zipatch = sdk2.parameters.Bool(
                "Apply zipatch and build arcadia", default=False)

            commit = sdk2.parameters.Bool(
                "Commit new policy to Arcadia", default=False)

        with sdk2.parameters.Group("token"):
            token = sdk2.parameters.YavSecret(
                "YA_TOKEN",
                default="sec-01fhsqqed5nvfp6zszzbghgrx8@ver-01fkzhmz903z2jpjbdnm8kfccg"
                )

        with sdk2.parameters.Group("Options"):
            excludes = sdk2.parameters.String(
                label='Comma-separated IDs of excluded vulnerabilities',
                default=None
            )
            whitelist = sdk2.parameters.String(
                label='Comma-separated IDs of whitelisted vulnerabilities',
                default=None
            )

    @cached_property
    def policy_path(self):
        return "build/rules/yadi.policy"

    def get_generate_cmd(self, binary_path, lang, excludes, whitelist):
        cmd = [
            binary_path,
            "--verbose"
        ]

        if excludes:
            cmd.extend(["--skips", excludes])

        if whitelist:
            cmd.extend(["--only", whitelist])

        cmd.extend([
            "policy",
            "generate",
            str(lang)
        ])

        return cmd

    def on_execute(self):
        import library.python.zipatch as zipatch
        # sdk2.Vault.data("YADI_ARCANUM_TOKEN") # there is no need while ChangesDetector only supports SVN API
        token = ""
        arcadia_url = ""

        if self.Parameters.strategy == "resource":
            assert self.Parameters.arcadia_url.startswith(
                "arcadia:/arc/trunk/arcadia@"), "Invalid Arcadia URL"
            arcadia_url = self.Parameters.arcadia_url
            patch = sdk2.ResourceData(self.Parameters.patch)
        elif self.Parameters.strategy == "build":
            arcadia_url = sdk2.vcs.svn.Arcadia.freeze_url_revision(self.Parameters.arcadia_url or sdk2.vcs.svn.Arcadia.trunk_url())
            with arcadiasdk.mount_arc_path(arcadia_url,
                                           use_arc_instead_of_aapi=False,
                                           arc_oauth_token=token) as arcadia:
                logger.info("Arcadia (%s) was mounted on path: %s",
                            arcadia_url, arcadia)
                language = self.Parameters.lang
                patch = self.build_zipatch_resource(zipatch, arcadia, language)

        if not self.Parameters.apply_zipatch:
            return

        # detect changes
        gg = ChangesDetector(
            self,
            description="Generate graph for YadiArcPolicy.\nparent_task={parent}, zipatch_resourse={resource}".format(
                parent=self.id,
                resource=patch.id,
            ),
            kill_timeout=3 * 60 * 60,
            notifications=self.Parameters.notifications,
            create_sub_task=True,
            arc_url_left=arcadia_url,
            patch_left=patch.skynet_id,
            arc_url_right=arcadia_url,
            patch_right=None,
            ya_token=self.Parameters.token,
            platforms=["linux", "mandatory"],
            targets='autocheck',
            distbuild_pool='//sas/users/yadi',
        )

        # specifying subtask parameters
        gg.save()
        gg.enqueue()

    def generate_policy(self, lang, arcadia_root, out_resource_path):
        logger.info("Generating yadi-arc policy with output to '%s'",
                    out_resource_path)

        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            cmd = self.get_generate_cmd(
                self.subtask_binary_path,
                lang, self.Parameters.excludes,
                self.Parameters.whitelist
                )
            env = self.get_env()

            with out_resource_path.open("wb") as fd:
                sdk2.helpers.subprocess.check_call(
                    cmd, shell=False, stdout=fd, stderr=pl.stderr, env=env, cwd=arcadia_root)

    def build_policy_resource(self, arcadia, language):
        policy = YadiArcPolicyResult(
            self, "YadiArcPolicyResult", sdk2.Path("yadi.policy"))
        self.generate_policy(language, arcadia, policy.path)
        return policy

    def build_zipatch_resource(self, zipatch, arcadia, language):
        policy_resource = self.build_policy_resource(arcadia, language)

        zp_resource = YadiArcPolicyZipatch(
            self, "YadiArcPolicyResultZipatch", sdk2.Path("yadi-policy.zipatch"))

        zp_resource_data = sdk2.ResourceData(zp_resource)

        logger.info("Generating yadi-arc policy zipatch to '%s'",
                    zp_resource.path)

        zw = zipatch.ZipatchWriter()
        zw.add_action("store_file", self.policy_path,
                      file=str(policy_resource.path))
        zw.save(str(zp_resource.path))

        zp_resource_data.ready()
        return zp_resource
