# -*- encoding: utf-8 -*-
import json
import logging

from sandbox import sdk2
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.common.apihelpers import get_last_resource_with_attribute
from sandbox.projects.yadrive.resources import YaDriveFunctional


class YaDriveFunctionalTask(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            PipEnvironment('startrek-client', '2.5', custom_parameters=['--upgrade-strategy only-if-needed']),
        ]
        cores = 1
        ram = 2048
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        drive_functional = sdk2.parameters.Resource(
            'Yandex.Drive functional tests',
            resource_type=YaDriveFunctional
        )
        drive_endpoint = sdk2.parameters.String(
            'Yandex.Drive backend endpoint',
            default='testing.carsharing.yandex.net'
        )
        drive_public_token = sdk2.parameters.Vault(
            'Public Drive token secret',
            required=True
        )
        drive_public_helper_token = sdk2.parameters.Vault(
            'Public Drive helper token secret',
            required=True
        )
        drive_private_token = sdk2.parameters.Vault(
            'Private Drive token secret',
            required=True
        )

        suites = sdk2.parameters.List('Suite')
        disable_suites = sdk2.parameters.List('Disable Suite')
        tests = sdk2.parameters.List('Suite::Test')
        ticket = sdk2.parameters.String('Star track release ticket')
        test_tags = sdk2.parameters.List('Test attributes')
        timeout_per_test = sdk2.parameters.Integer('Timeout Per test')

        notifications = [
            sdk2.Notification(
                [ctt.Status.Group.FINISH],
                ['keksite', 'lyakich', 'lillval'],
                ctn.Transport.TELEGRAM
            )
        ]

    def on_execute(self):
        binary_resource = self.Parameters.drive_functional
        if not binary_resource:
            binary_resource = get_last_resource_with_attribute(YaDriveFunctional)

        assert binary_resource
        binary = sdk2.ResourceData(binary_resource)

        private_token = self.Parameters.drive_private_token
        public_token = self.Parameters.drive_public_token
        public_helper_token = self.Parameters.drive_public_helper_token
        assert private_token
        assert public_token
        cmd = [
            str(binary.path),
            '-c', public_token.data(),
            '-ch', public_helper_token.data(),
            '-s', private_token.data(),
        ]
        if self.Parameters.drive_endpoint:
            cmd.append('--endpoint')
            cmd.append(self.Parameters.drive_endpoint)
        if self.Parameters.suites:
            cmd.append('--suites')
            for suite in self.Parameters.suites:
                cmd.append(suite)
        if self.Parameters.disable_suites:
            cmd.append('-ds')
            for disable_suite in self.Parameters.disable_suites:
                cmd.append(disable_suite)
        if self.Parameters.tests:
            cmd.append('--tests')
            for test in self.Parameters.tests:
                cmd.append(test)

        if self.Parameters.test_tags:
            cmd.append('--tags')
            for test_tag in self.Parameters.test_tags:
                cmd.append(test_tag)

        if self.Parameters.timeout_per_test:
            cmd.append('-to')
            cmd.append(str(self.Parameters.timeout_per_test))

        log = sdk2.helpers.ProcessLog(self, logger='functional')
        with log:
            sdk2.helpers.subprocess.call(cmd, stdout=log.stdout, stderr=log.stderr)

        passed = 0
        skipped = 0
        failed = 0
        failed_tests = []
        skipped_tests = []

        lines = log.stdout.path.open().readlines()
        for line in lines:
            try:
                ev = json.loads(line)
                typ = ev['type']
                source = ev['source']
                message = ev['message']
                logging.info('event={}, source={}, message={}'.format(typ, source, message))

                if typ == 'pass':
                    passed += 1
                if typ == 'skip':
                    skipped += 1
                    skipped_tests.append(source)
                if typ == 'failure':
                    failed += 1
                    failed_tests.append(source)
            except Exception as e:
                logging.error("cannot parse line {} as event: {}".format(line, e))

        self.set_info('{} tests passed'.format(passed))
        self.set_info(self.log_path())
        if skipped > 0:
            self.set_info('{} tests skipped:\n{}'.format(
                skipped,
                '\n'.join(skipped_tests)
            ))
        if failed > 0:
            self.set_info('{} tests failed:\n{}'.format(
                failed,
                '\n'.join(failed_tests)
            ))
            raise Exception('several tests failed')

        if self.Parameters.ticket and failed == 0:
            st_token = sdk2.yav.Secret("sec-01e2g8sg2mbr3sf662yanzwgpj").data()['prime-token']

            from startrek_client import Startrek
            st_client = Startrek(useragent='ya.drive', token=st_token)
            issue = st_client.issues[self.Parameters.ticket]
            issue_tags = issue.tags
            new_tag = "autotest_{}".format(self.Parameters.drive_endpoint)
            if new_tag not in issue_tags:
                self.log_resource.ttl = 365
                issue_tags.append(new_tag)
                issue.update(tags=issue_tags)
                issue.remotelinks.create(origin='ru.yandex.sandbox', relationship="relates", key=self.id)
                msg = "Приемочные тесты пройдены на окружении {} тут: https://sandbox.yandex-team.ru/task/{} .\n" \
                      "Логи: https://proxy.sandbox.yandex-team.ru/{}".format(self.Parameters.drive_endpoint,
                                                                             self.id,
                                                                             self.log_resource.id)
                comment = issue.comments.create(text=msg, summonees=["dbarbashin", "lyakich"])
