# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import link_builder
from sandbox.common.types.task import Status
from sandbox.common.types import resource


def get_sandbox_task_url(task_id):
    return 'https://sandbox.yandex-team.ru/task/%s/view' % task_id


class DriveBackendAutotests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        push_tasks_resource = True

        with sdk2.parameters.Group("DriveAutotests"):
            send_report_to_st_issue = sdk2.parameters.String(
                label='Send report to ST issue',
                description='Issue key; tests results will be posted as comment',
            )

        with sdk2.parameters.Output():
            are_tests_ok = sdk2.parameters.Bool('Are tests ok?')
            allure_report_url = sdk2.parameters.String('Allure report url')

    def run_subtask(self):
        task_class = sdk2.Task['YA_MAKE_2']
        subtask = task_class(
            self,
            description='Run drive backend autotests',
            targets='drive/tests/functional_pytest',
            test=True,
            allure_report=True,
            allure_report_ttl=365,
            use_aapi_fuse=True,
            use_arc_instead_of_aapi=True,
            env_vars="YAV_TOKEN=\'$(vault:file:EXTMAPS:drive_functional_tests)\'",
            test_type_filter='pytest',
            test_size_filter='large',
            test_log_level='debug'
        )
        subtask.enqueue()
        self.Context.subtask_id = subtask.id
        self.Context.save()
        raise sdk2.WaitTask(subtask.id, Status.Group.FINISH + Status.Group.BREAK)

    @sdk2.report(title='Tests result', label='tests_result')
    def report(self):
        title_style = 'style="line-height: 2; font-size: 20px"'
        link_style = 'style="font-size: 16px"'

        if not self.Context.tests_stats:
            return '<p %s><b>Тесты ещё не завершились</b></p>' % title_style
        else:
            total_tests = self.Context.tests_stats['total']
            failed_tests = self.Context.tests_stats['failed']
            skipped_tests = self.Context.tests_stats['skipped']

            if failed_tests:
                message = '<font color="red">Сломалось %s тестов из %s</font>' % (failed_tests, total_tests)
            else:
                message = '<font color="green">Прошло %s тестов, пропущено %s</font>' % (
                    total_tests - skipped_tests,
                    skipped_tests,
                )
            return '<p %s><b>%s</b></p> <p %s><a href="%s" target="_self">Allure-отчёт</a></p>' % (
                title_style,
                message,
                link_style,
                self.Parameters.allure_report_url,
            )

    def send_report_to_st_issue(self, issue_key, allure_report_url):
        msg = '(({} Запуск)) python-автотестов в  QA:' \
              ' \n\n<{{Отчет' \
              ' \n\n/iframe/(src={} width=100% height=900 frameborder= scrolling= allowfullscreen=true)}}>' \
            .format(get_sandbox_task_url(self.id), allure_report_url)
        self.post_comment_to_st(issue_key, msg, remote_link=self.id)

    @staticmethod
    def post_comment_to_st(issue_key, comment, remote_link=None):
        st_token = sdk2.yav.Secret("sec-01e2g8sg2mbr3sf662yanzwgpj").data()['prime-token']

        from startrek_client import Startrek
        st_client = Startrek(useragent='ya.drive', token=st_token)
        issue = st_client.issues[issue_key]
        issue_tags = issue.tags
        new_tag = "autotest_qa"
        if new_tag in issue_tags:
            return
        issue_tags.append(new_tag)
        issue.update(tags=issue_tags)
        if remote_link:
            issue.remotelinks.create(origin='ru.yandex.sandbox', relationship="relates", key=remote_link)
        st_client.issues[issue_key].comments.create(text=comment, params={'isAddToFollowers': False})

    def check_subtask_result(self):
        subtask = sdk2.Task.find(id=self.Context.subtask_id).first()
        allure_report_resource = sdk2.Resource['ALLURE_REPORT'].find(task=subtask).first()
        if not allure_report_resource:
            raise errors.TaskError('No allure report found (seems like subtask has failed)')
        allure_report_url = '%s/index.html' % allure_report_resource.http_proxy
        self.Parameters.allure_report_url = allure_report_url

        msg = (
            'Subtask {subtask} status: {task_status}\n'
            '<a href="{allure_report_url}">Allure report</a>'
        ).format(
            subtask=link_builder.task_link(subtask.id, plain=False),
            task_status=subtask.status,
            allure_report_url=allure_report_url,
        )
        self.set_info(info=msg, do_escape=False)

        st_issue_key = self.Parameters.send_report_to_st_issue

        if st_issue_key and subtask.status == Status.SUCCESS:
            self.send_report_to_st_issue(
                issue_key=st_issue_key,
                allure_report_url=allure_report_url,
            )

        # Если subtask.status == Status.SUCCESS - значит, все тесты прошли.
        # Иначе - сломалась часть тестов или задача целиком.
        are_tests_ok = subtask.status == Status.SUCCESS
        self.Parameters.are_tests_ok = are_tests_ok

        if not are_tests_ok:
            raise errors.TaskError('Some tests have failed')

    def on_create(self):
        if not self.Requirements.tasks_resource:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                owner='EXTMAPS',
                state=resource.State.READY,
                attrs={'task_type': 'DRIVE_BACKEND_AUTOTESTS', 'auto_deploy': True},
            ).first()

    def on_execute(self):
        with self.memoize_stage.run_subtask:
            self.run_subtask()

        self.check_subtask_result()
