import json
from sandbox import common
from sandbox import sdk2
from sandbox.projects.common.nanny import nanny

ROLES_TO_COMMANDS_PATH = 'pumpkin.roles_to_commands'

ROLES_TO_COMMANDS_SCHEMA = {
    "type" : "array",
    "items": {
        "type": "object",
        "properties" : {
            "roles" : {
                "type": "array",
                "items": {
                    "type": "string"
                }
            },
            "commands": {
                "type": "array",
                "items": {
                    "type": "string"
                }
            }
        },
        "required": ["roles", "commands"]
    }
}

ROLES_USERS_SCHEMA = {
    "type": "array",
    "items": {
        "type": "object",
        "properties": {
            "username": {
                "type": "string"
            }
        },
        "required": ["username"]
    }
}

CAR_LIST_SCHEMA = {
    "type": "array",
    "items": {
        "type": "object",
        "properties": {
            "imei": {
                "type": "string"
            },
            "number": {
                "type": "string"
            },
            "id": {
                "type": "string"
            }
        },
        "required": ["imei", "number", "id"]
    }
}


class YaDrivePumpkinDataFetcher(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        drive_token_name = sdk2.parameters.String("Drive staff token (Sandbox Vault)")
        with sdk2.parameters.RadioGroup("Deploy to") as cluster:
            cluster.values["testing"] = cluster.Value("testing")
            cluster.values["stable"] = cluster.Value("stable", default=True)

    def on_execute(self):
        from sandbox.projects.yadrive.resources import \
            YaDrivePumpkinUsersPermissions, \
            YaDrivePumpkinCarsData
        from jsonschema import validate
        from drive.backend.api import client as api
        drive_token = sdk2.Vault.data(self.owner, self.Parameters.drive_token_name)

        if self.Parameters.cluster == "testing":
            url = "https://testing.carsharing.yandex.net"
        else:
            url = "https://admin.carsharing.yandex.net"

        client = api.BackendClient(endpoint=url, private_token=drive_token, require_public_token=False)
        settings_response = client.list_settings(prefix=ROLES_TO_COMMANDS_PATH)
        if settings_response is None:
            raise common.errors.TaskFailure("Can't fetch ACL")
        if settings_response[0]['setting_key'] != ROLES_TO_COMMANDS_PATH:
            raise common.errors.TaskFailure(f"setting_key is {settings_response[0]['setting_key']}, expected {ROLES_TO_COMMANDS_PATH}")
        roles_to_commands = json.loads(settings_response[0]['setting_value'])
        validate(roles_to_commands, ROLES_TO_COMMANDS_SCHEMA)

        users_permissions_dict = dict()

        for batch in roles_to_commands:
            for role in batch['roles']:
                role_to_users = client.list_role_users(role)
                if role_to_users is None:
                    raise common.errors.TaskFailure("Can't fetch users per role: {}".format(role))
                validate(role_to_users, ROLES_USERS_SCHEMA)
                for user in role_to_users:
                    username = user['username']
                    if username not in users_permissions_dict:
                        users_permissions_dict[username] = {'user_login' : username,
                                                            'allowed_commands': []}
                    users_permissions_dict[username]['allowed_commands'].extend(batch['commands'])

        users_permissions_list = [{'user_login' : val['user_login'], 'allowed_commands' : list(set(val['allowed_commands']))} for val in users_permissions_dict.values()]
        users_permissions_resource = sdk2.ResourceData(YaDrivePumpkinUsersPermissions(self, "Users' permissions", "users_permissions.json"))
        users_permissions_resource.path.write_bytes(json.dumps(users_permissions_list).encode('utf-8'))
        users_permissions_resource.ready()

        car_list_response = client.raw_list_cars()
        if not car_list_response:
            raise common.errors.TaskFailure("Can't fetch users per role: {}".format(role))
        car_list = car_list_response['cars']
        validate(car_list, CAR_LIST_SCHEMA)
        cars_data_list = []
        for car_data in car_list:
            if len(car_data['imei'])>0:
                cars_data_list.append({'imei': car_data['imei'], 'id': car_data['id'], 'number': car_data['number']})
        cars_data_resource = sdk2.ResourceData(YaDrivePumpkinCarsData(self, "Cars data", "cars_data.json"))
        cars_data_resource.path.write_bytes(json.dumps(cars_data_list).encode('utf-8'))
        cars_data_resource.ready()


class YaDrivePumpkinUpdater(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        push_tasks_resource = True
        drive_token_name = sdk2.parameters.String("Drive staff token (Sandbox Vault)")
        with sdk2.parameters.RadioGroup("Deploy to") as cluster:
            cluster.values["testing"] = cluster.Value("testing")
            cluster.values["stable"] = cluster.Value("stable", default=True)

    def _create_child_tasks(self):
        child = YaDrivePumpkinDataFetcher(
            self,
            description="Child of {}".format(self.id),
            owner=self.owner
        )
        child.Parameters.drive_token_name = self.Parameters.drive_token_name
        child.Parameters.cluster = self.Parameters.cluster
        child.save()
        child.enqueue()
        return child.id

    def on_execute(self):
        if not self.Context.subtask_id:
            subtask_id = self._create_child_tasks()  # task objects or task ids
            # Save ids to context to process them after waking up
            self.Context.subtask_id = subtask_id
            raise sdk2.WaitTask([subtask_id], common.types.task.Status.Group.FINISH | common.types.task.Status.Group.BREAK, wait_all=True)
        self.server.release(task_id=self.Context.subtask_id, type=self.Parameters.cluster, subject='Release')
