# -*- encoding: utf-8 -*-
import json
import logging

from sandbox import sdk2
from sandbox.common.types import notification as ctn
from sandbox.projects.common.apihelpers import get_last_released_resource

from sandbox.projects.yadrive.resources import YaDriveCreateAccounts

recipients = ["dbarbashin", "tretyakovaan"]

class AccountsResource(sdk2.Resource):
    """ Resource with list of test accounts"""


class YaDriveTestAccountsGenerator(sdk2.Task):

    class Parameters(sdk2.Parameters):

        drive_accounts = sdk2.parameters.Resource(
            'Yandex.Drive accounts creator binary',
            resource_type=YaDriveCreateAccounts
        )
        drive_private_token = sdk2.parameters.Vault(
            'Private Drive token secret',
            required=True
        )
        drive_public_token = sdk2.parameters.Vault(
            'Public Drive token secret',
            required=True
        )
        tus_token = sdk2.parameters.YavSecret(
            'TUS token',
            required=True
        )
        roles = sdk2.parameters.String('roles_id')

        status = sdk2.parameters.String('Status',
                                        required=True,
                                        choices=[('onboarding', 'onboarding'), ('active', 'active')],
                                        ui=sdk2.parameters.String.UI('select'))
        source = sdk2.parameters.String('Source',
                                        required=True,
                                        choices=[('qa', 'qa'), ('testing', 'testing')],
                                        ui=sdk2.parameters.String.UI('select'))

        accounts_number = sdk2.parameters.Integer("number of accounts", required=True)

    def on_execute(self):
        binary_resource = self.Parameters.drive_accounts
        if not binary_resource:
            binary_resource = get_last_released_resource(YaDriveCreateAccounts)

        assert binary_resource
        binary = sdk2.ResourceData(binary_resource)

        private_token = self.Parameters.drive_private_token
        public_token = self.Parameters.drive_public_token
        tus_token = self.Parameters.tus_token

        assert private_token
        assert public_token
        cmd = [
            str(binary.path),
            '--public-token', public_token.data(),
            '--private-token', private_token.data(),
            'create_accounts',
            '--tus-token', tus_token.data()['drive-tus-token'],
        ]

        if self.Parameters.status:
            cmd.append('--status')
            cmd.append(self.Parameters.status)
        if self.Parameters.roles:
            cmd.append('-r')
            cmd.append(self.Parameters.roles)
        if self.Parameters.accounts_number:
            cmd.append('-n')
            cmd.append(str(self.Parameters.accounts_number))
        if self.Parameters.source:
            cmd.append('--source')
            cmd.append(str(self.Parameters.source))
        resource = AccountsResource(self, "List of accounts", "accounts.txt")
        resource_data = sdk2.ResourceData(resource)
        log = sdk2.helpers.ProcessLog(self, logger='functional')
        with log:
            sdk2.helpers.subprocess.call(cmd, stdout=log.stdout, stderr=log.stderr)
        accounts_txt = ''
        lines = log.stdout.path.open().readlines()
        logging.info(lines)
        for line in lines:
            try:
                ev = json.loads(line)
                username = ev['username']
                password = ev['password']
                status = ev['status']
                accounts_txt += ('{};{};{};\"Да\"\n'.format(username, password, status))
                logging.info(accounts_txt)
                logging.info(accounts_txt)
            except Exception as e:
                logging.error('cannot parse line {} as event: {}'.format(line, e))

        resource_data.path.write_bytes(accounts_txt)
        resource_data.ready()

        self.set_info("{}".format(resource.http_proxy))

        self.server.notification(
            body="{}".format(resource.http_proxy),
            recipients=recipients,
            transport=ctn.Transport.TELEGRAM
        )
