# -*- coding: utf-8 -*-
from sandbox.projects import resource_types
import sandbox.projects.yane.common as yane
from sandbox.projects.common.utils import check_subtasks_fails
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter, LastReleasedResource, SandboxInfoParameter, SandboxFloatParameter
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.yane.TotalLearn import MODELS_DEFAULT

COMMIT_DESTINATIONS = {
    'commit_result_to_ES_prod': 'search/wizard/entitysearch/data/ner',
    'commit_result_to_ES_test': 'search/wizard/entitysearch/data/ner_test',
    'commit_result_to_begemot': 'junk/yourmary/temp_ner/begemot',
    'yt_path': '//home/dict/yourmary/temp_ner'
}


class YaneBuildAndLearn(yane.YaneTaskBase):
    """
        Learns all models
    """
    type = 'YANE_BUILD_AND_LEARN'

    execution_space = 120 * 1024

    class SourceOntoDB(SandboxStringParameter):
        name = 'source_ontodb'
        description = 'Source Onto DB table'
        default_value = 'home/dict/ontodb/ver/main/production/all_cards_final'
        group = yane.GROUP_IN
        required = True

    class TargetFolder(SandboxStringParameter):
        name = 'target'
        description = 'Target MR folder prefix'
        default_value = 'home/dict/yane/db.NNN'
        group = yane.GROUP_OUT
        required = True

    class MatrixnetBinary(LastReleasedResource):
        name = 'matrixnet'
        description = 'Matrixnet binary (optional, leave blank to build from trunk)'
        resource_type = resource_types.MATRIXNET_EXECUTABLE
        required = False

    class CommitTo(SandboxInfoParameter):
        name = "commit_to_descr"
        description = "On success commit result to:"

    class CommitToProd(SandboxBoolParameter):
        name = 'commit_result_to_ES_prod'
        default_value = True
        description = "prod NER slot"

    class CommitToTest(SandboxBoolParameter):
        name = 'commit_result_to_ES_test'
        description = "test NER slot"

    class CommitToBegemot(SandboxBoolParameter):
        name = 'commit_result_to_begemot'
        default_value = True
        description = "begemot"

    class FMeasureThresholdDescr(SandboxInfoParameter):
        name = "fmeasure_descr"
        description = "Commit models if all fmeasures are greater:"

    class FMeasureThresholdRuQuery(SandboxFloatParameter):
        name = 'fmeasure_threshold_ru_query'
        description = 'Ru, query'
        default_value = 0.865
        required = True

    class FMeasureThresholdRuText(SandboxFloatParameter):
        name = 'fmeasure_threshold_ru_text'
        description = 'Ru, text'
        default_value = 0.87
        required = True

    class FMeasureThresholdRuVideo(SandboxFloatParameter):
        name = 'fmeasure_threshold_ru_video'
        description = 'Ru, video'
        default_value = 0.83
        required = True

    class EnableBorsches(SandboxBoolParameter):
        name = 'with_borsches'
        description = 'Turn on Borsches'
        default_value = False
        sub_fields = {
            'false': ['commit_to_descr', 'commit_result_to_ES_prod', 'commit_result_to_ES_test', 'commit_result_to_begemot',
                      'fmeasure_descr', 'fmeasure_threshold_ru_query', 'fmeasure_threshold_ru_text', 'fmeasure_threshold_ru_video']
        }

    class YtUploadPath(SandboxStringParameter):
        name = 'yt_upload_path'
        description = 'YT path to upload NER data'
        default_value = COMMIT_DESTINATIONS['yt_path']

    input_parameters = [SourceOntoDB, TargetFolder, EnableBorsches, CommitTo, CommitToProd, CommitToTest, CommitToBegemot, YtUploadPath,
                        FMeasureThresholdDescr, FMeasureThresholdRuQuery, FMeasureThresholdRuText, FMeasureThresholdRuVideo]\
                       + yane.get_base_params().params + yane.get_mr_params().params \
                       + [MatrixnetBinary]

    @yane.run_once
    def prepare_data(self):
        params = {
            'svn_url': self.ctx['svn_url'],
            'arcadia_patch': self.ctx.get('arcadia_patch'),
            'tools': self.ctx['tools'],
            'mr_server': self.ctx.get('mr_server'),
            'notify_via': '',
            'notify_if_finished': '',
            'notify_if_failed': self.author,
            'source_ontodb': self.ctx['source_ontodb'],
            'execution_space': self.execution_space,
            'target': '%s.%s' % (self.ctx['target'], 'ru'),
            'with_borsches': self.ctx['with_borsches']
        }
        subtask = self.create_subtask('YANE_PREPARE_DATA',
                                        '%s, %s' % (self.descr, 'ru'),
                                        params)
        self.ctx['ru_data_task'] = subtask.id

        self.ctx['child_tasks'] = [subtask.id]

    @yane.run_once
    def total_learn(self):
        params = {
            'svn_url': self.ctx['svn_url'],
            'arcadia_patch': self.ctx.get('arcadia_patch'),
            'tools': self.ctx['tools'],
            'matrixnet': self.ctx['matrixnet'],
            'russian_data': channel.sandbox.get_task(self.ctx['ru_data_task']).ctx['data_id'],
            'notify_via': '',
            'notify_if_finished': '',
            'notify_if_failed': self.author,
            'with_borsches': self.ctx['with_borsches']
        }

        subtask = self.create_subtask('YANE_TOTAL_LEARN',
                                      self.descr,
                                      params)
        self.ctx['child_tasks'].append(subtask.id)
        self.ctx['total_learn_task'] = subtask.id

    @yane.run_once
    def commit_result(self):
        paths_es = []
        for option in ['commit_result_to_ES_prod', 'commit_result_to_ES_test']:
            if self.ctx[option]:
                paths_es.append(COMMIT_DESTINATIONS[option])
        path_begemot = '' if not self.ctx['commit_result_to_begemot'] else COMMIT_DESTINATIONS['commit_result_to_begemot']
        if paths_es or path_begemot:
            params = {
                'svn_url': self.ctx['svn_url'],
                'tools': self.ctx['tools'],
                'mr_server': self.ctx.get('mr_server'),
                'svn_paths_es': ' '.join(paths_es),
                'svn_paths_begemot': path_begemot,
                'total_learn_task': self.ctx['total_learn_task'],
                'yt_upload_path': self.ctx['yt_upload_path'],
            }
            for lang in ['ru', 'tr']:
                for model in MODELS_DEFAULT:
                    opt = '_'.join(['fmeasure_threshold', lang, model])
                    if opt in self.ctx:
                        params[opt] = self.ctx[opt]
            subtask = self.create_subtask('YANE_BUILD_AND_COMMIT',
                                          self.descr,
                                          params)
            self.ctx['child_tasks'].append(subtask.id)

    def __init__(self, task_id=0):
        yane.YaneTaskBase.__init__(self, task_id)
        self.ctx['kill_timeout'] = 8 * 60 * 60

    def do_execute(self):
        self.prepare_data()
        self.build_matrixnet()
        self.total_learn()
        if not self.ctx['with_borsches']:
            self.commit_result()

        if 'childs' in self.ctx:  # проснулись после ожидания child_tasks (wait_all_tasks_stop_executing)
            check_subtasks_fails()
        else:
            self.ctx['childs'] = True
            self.wait_all_tasks_stop_executing(self.ctx['child_tasks'])


__Task__ = YaneBuildAndLearn
