# -*- coding: utf-8 -*-

import os.path

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
import sandbox.projects.yane.common as yane
from sandbox.sandboxsdk.parameters import SandboxStringParameter, LastReleasedResource
from sandbox.sandboxsdk.svn import Arcadia

RELURLS_IDS_TRIE_NAME = 'object.relurls.trie'


class YaneParseRelurls(yane.YaneTaskBase):
    """
        Parses related URLs and generates relurls trie
    """

    type = 'YANE_PARSE_RELURLS'

    # tools - 5Gb, resource - 1Gb
    execution_space = 20 * 1024
    client_tags = ctc.Tag.LINUX_PRECISE

    class Config(LastReleasedResource):
        name = 'config'
        description = 'Config'
        resource_type = resource_types.YANE_CONFIG
        group = yane.GROUP_IN

    class ObjectsTrie(LastReleasedResource):
        name = 'trie'
        description = 'Objects external IDs trie'
        resource_type = resource_types.OTHER_RESOURCE
        group = yane.GROUP_IN

    class Target(SandboxStringParameter):
        name = 'target'
        description = 'Target table'
        default_value = 'home/dict/yane/db.NNN/relurls/urls'
        group = yane.GROUP_OUT
        required = True

    input_parameters = \
        yane.get_base_params().params + \
        [Config, ObjectsTrie, Target] + \
        yane.get_mr_params().params

    def __init__(self, task_id=0):
        yane.YaneTaskBase.__init__(self, task_id)
        self.ctx['kill_timeout'] = 6 * 60 * 60

    def on_enqueue(self):
        resource = self._create_resource('Relurls IDs trie',
                                         RELURLS_IDS_TRIE_NAME,
                                         resource_types.OTHER_RESOURCE,
                                         arch='Any')
        self.ctx['trie_id'] = resource.id
        yane.YaneTaskBase.on_enqueue(self)

    @yane.run_once
    def _run_relurlsextractor(self):
        yane_config = self.get_config('config')
        cfg_svn_path, cfg_name = os.path.split(yane_config['relurls']['config'])
        if not os.path.exists('cfg'):
            os.mkdir('cfg')
            Arcadia.checkout(self.get_svn_path(cfg_svn_path), path='cfg')

        self.run_tool('relurlsextractor',
                      ['-s', self.ctx['mr_server'],
                       '-i', yane_config['clicks']['source'],
                       '-o', self.ctx['target'],
                       '-t', self.sync_resource(self.ctx['trie']),
                       '-c', os.path.join('cfg', cfg_name),
                       ], self.get_mr_env())

    @yane.run_once
    def _run_trie_compiler(self):
        self.compile_trie(self.ctx['target'], RELURLS_IDS_TRIE_NAME)
        self.mark_resource_ready(self.ctx['trie_id'])

    def do_execute(self):
        self._run_relurlsextractor()
        self._run_trie_compiler()


__Task__ = YaneParseRelurls
