# -*- coding: utf-8 -*-
import logging
import time

from sandbox import sdk2
from sandbox.projects.yappy.api import YappyApi
from sandbox.projects.yappy import configs

LOGGER = logging.getLogger('yappy')


def _collect_beta_templates_configs():
    mapping = {}

    for file_name in dir(configs):
        if not file_name.startswith('_'):
            module = getattr(configs, file_name)
            LOGGER.info('find configs in module %s', module)

            for item in dir(module):
                cfg = getattr(module, item)

                if isinstance(cfg, configs.main_configs.BetaTemplateConfig):
                    mapping[cfg.template_name] = cfg

    return mapping


class YappyBetaCreationTimeoutError(Exception):
    pass


class CreateYappyBeta(sdk2.Task):
    configs = _collect_beta_templates_configs()

    class Parameters(sdk2.task.Parameters):
        template_name = sdk2.parameters.String('Template name', required=True)
        beta_suffix = sdk2.parameters.String('Beta name suffix', required=True)
        resources = sdk2.parameters.Dict('Resource local path (or alias) to resource id')
        beta_consistency_timeout = sdk2.parameters.Integer('Beta consistency timeout (sec)', default_value=1800)
        update_if_exist = sdk2.parameters.Bool('Update beta if exists', default=False)

    def on_yappy_beta_status(self, status):
        pass

    def on_execute(self):
        if self.Parameters.template_name not in self.configs:
            raise ValueError('can\'t find template {} in configs dir'.format(self.Parameters.template_name))

        token = sdk2.Vault.data('SEARCH-RELEASERS', 'yappy_auth_token')
        api = YappyApi(oauth_token=token)

        beta_name = '{}-{}'.format(self.Parameters.template_name, self.Parameters.beta_suffix)
        LOGGER.info('check beta %s exist', beta_name)

        if not self.Parameters.update_if_exist and api.beta_exists(payload={'name': beta_name}).get('exists'):
            raise ValueError('beta {} is already exist'.format(beta_name))

        beta_cfg = self.configs[self.Parameters.template_name]
        payload = {
            'template_name': self.Parameters.template_name,
            'suffix': self.Parameters.beta_suffix,
            'update_if_exist': self.Parameters.update_if_exist,
            'patches': [],
        }

        LOGGER.info('create beta payload #1 %s', payload)

        for beta_component in beta_cfg.components:
            payload['patches'].append({
                'componentId': beta_component.id,
                'patch': {
                    'ignoreParentInstanceSpec': beta_component.patch.ignore_instance_spec,
                    'parentExternalId': beta_component.patch.parent_external_id,
                    'resources': [],
                    'copyCoredumpPolicy': beta_component.patch.copy_coredump_policy
                }
            })

            LOGGER.info('create beta payload #2 %s, bc_id %s', payload, beta_component.id)

            for resource in beta_component.patch.resources:
                resource_id = resource.alias or resource.local_path

                if isinstance(resource, configs.main_configs.ResourceConfig) and resource_id in self.Parameters.resources:
                    payload['patches'][-1]['patch']['resources'].append({
                        'manageType': resource.manage_type,
                        'localPath': resource.local_path,
                        'storage': resource.storage,
                        'sandboxResourceId': self.Parameters.resources[resource_id]
                    })

                if isinstance(resource, configs.main_configs.StaticContentConfig):
                    if resource_id in self.Parameters.resources:
                        resource.content = self.Parameters.resources[resource_id]

                    payload['patches'][-1]['patch']['resources'].append({
                        'manageType': 'STATIC_CONTENT',
                        'localPath': resource.local_path,
                        'content': resource.content
                    })
                LOGGER.info('create beta payload #3 %s, resource_id %s', payload, resource_id)

        LOGGER.info('create beta with payload %s', payload)
        api.create_beta_from_template(payload=payload)

        end_time = time.time() + self.Parameters.beta_consistency_timeout
        sleep_interval = 60

        while time.time() < end_time:
            sleep_interval = min(sleep_interval, max(end_time - time.time(), 0))
            LOGGER.info('wait beta consistent, sleep %s', sleep_interval)
            time.sleep(sleep_interval)

            get_beta_status_response = api.get_beta_status(payload={'name': beta_name})
            LOGGER.info('got beta status response %s', get_beta_status_response)

            beta_status = get_beta_status_response.get('state', {}).get('status')
            self.on_yappy_beta_status(beta_status)

            if beta_status == 'CONSISTENT':
                return

        raise YappyBetaCreationTimeoutError('Beta consistency timeout')
