import os

import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.projects.yasm import resource_types
from sandbox.common import fs
from sandbox.sandboxsdk import paths
from subprocess import PIPE


class ClickhouseVersionParameter(sdk_parameters.SandboxStringParameter):
    name = 'clickhouse_version'
    description = 'Clickhouse version'
    required = True


DEB_SHORT_NAME = "clickhouse-server-base"
DEB_COMMON_TEMPLATE = "clickhouse-server-common_{version}_amd64.deb"
DEB_BASE_TEMPLATE = "{name}_{version}_amd64.deb"
METRIKA_REPO_URL = "http://yandex-precise.dist.yandex.ru"

TARGET_FILES = [
    "./usr/bin/clickhouse-server",
]

TEMP_DIR = "temp"
RESULT_DIR = "dist"
CLICKHOUSE_SOURCES = "./clickhouse_sources.list"
CLICKHOUSE_ADDREPO = "deb http://yandex-precise.dist.yandex.ru/yandex-precise stable/amd64/"


class BuildClickhouseYasmPackage(sdk_task.SandboxTask):
    """
    Build latest Clickhouse version from sources.
    """

    type = "BUILD_CLICKHOUSE_YASM"

    resource_type = str(resource_types.CLICKHOUSE_YASM_BINARY)
    input_parameters = [ClickhouseVersionParameter]
    client_tags = ctc.Tag.Group.LINUX

    def on_execute(self):
        fs.make_folder(TEMP_DIR, delete_content=True)
        fs.make_folder(RESULT_DIR, delete_content=True)

        with fs.WorkDir(TEMP_DIR):
            clickhouse_version = self.ctx["clickhouse_version"]
            base_name = DEB_BASE_TEMPLATE.format(name=DEB_SHORT_NAME, version=clickhouse_version)

            # NOTE: Some configs are in 'common' deb
            # common_name = DEB_COMMON_TEMPLATE.format(
            #     version=clickhouse_version)

            with open(CLICKHOUSE_SOURCES, 'w+') as f:
                f.write(CLICKHOUSE_ADDREPO)

            apt_cache_out, apt_cache_err = sdk_process.run_process(
                ['/usr/bin/apt-cache', 'show', '{}={}'.format(DEB_SHORT_NAME, clickhouse_version), '-o', 'Dir::Etc::SourceList={}'.format(CLICKHOUSE_SOURCES)],
                stdout=PIPE,
                stderr=PIPE,
                log_prefix='get_package_url'
            ).communicate()

            real_url = "{}{}".format(METRIKA_REPO_URL, apt_cache_out.split()[1])

            with self.current_action("Downloading debs"):
                fs.fetch_file_via_http(real_url,
                                       base_name)

            with self.current_action("Unpack debs"):
                for package in [base_name]:
                    sdk_process.run_process(
                        ["dpkg", "-x", package, "."],
                        log_prefix="unpack_{}".format(package)
                    )

            with self.current_action("Copy target files"):
                for target_file in TARGET_FILES:
                    paths.copy_path(target_file,
                                    os.path.join("..", RESULT_DIR))

        with self.current_action("Create resource"):
            self.create_resource(
                description='Clickhouse binary version {}'.format(
                    clickhouse_version),
                resource_path=RESULT_DIR,
                resource_type=self.resource_type,
            )


__Task__ = BuildClickhouseYasmPackage
