# coding: utf-8

import textwrap

from sandbox.common.types.client import Tag

import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.paths as sdk_paths

from sandbox.projects.yasm import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.build.BuildForAllTask import DoNotRemoveResourcesParameter
from sandbox.sandboxsdk.process import run_process


class YasmAgent(sdk_parameters.LastReleasedResource):
    name = 'yasm_agent'
    description = 'YASM agent resource'
    resource_type = resource_types.YASMCORE


class BuildSkycoreYasmAgent(sdk_task.SandboxTask):
    type = 'BUILD_SKYCORE_YASM_AGENT'
    client_tags = Tag.GENERIC | Tag.SKYNET
    input_parameters = [
        YasmAgent,
        DoNotRemoveResourcesParameter
    ]
    execution_space = 4 * 1024

    def checkout(self):
        resource_id = utils.get_or_default(self.ctx, YasmAgent)
        tar_gz = self.sync_resource(resource_id)
        sdk_paths.make_folder('yasmcore', delete_content=True)
        run_process(['tar', '--extract', '--file', tar_gz, '--directory', 'yasmcore', '--verbose'], log_prefix='unpack_yasmcore')

    def write_scripts(self):
        sdk_paths.make_folder('yasmcore/bin', delete_content=True)

        command_to_execute = " ".join((
            'yasmagent/run.py',
            '--fg',
            '--config', '"${CURDIR}/yasmagent/agent.conf"',
            '--http-port', '${cfg:skynet.services.yasm:agent_port}',
            '--yasmutil-dir', '"${CURDIR}/yasmagent/lib"',
            '--type-conf-dir', '"${CURDIR}/yasmagent/CONF"',
            '--log-dir', '"${RUNDIR}/"',
            '--pid-file', '"${RUNDIR}/agent.pid"',
            '--keep-previous',
            '--push-enabled',
            '--push-http-port', '${cfg:skynet.services.yasm:agent_push_port}',
            '--push-limit-tags-combinations', '100',
            '--push-limit-total-values', '1000',
            '--instance-type', 'issagent',
            '--instance-type', 'porto',
            '--cutted-common'
        ))
        with open('yasmcore/yasmagent.scsd', 'w') as descr:
            descr.write(
                textwrap.dedent(
                    '''
                    conf: []
                    require: []
                    exec:
                        - {command_to_execute}
                    check: /usr/bin/env python bin/check "${{RUNDIR}}/agent.pid"
                    stop: /usr/bin/env python bin/stop ${{cfg:skynet.services.yasm:agent_port}} "${{RUNDIR}}/agent.pid"
                    porto: no
                    user: nobody
                    cgroup: null
                    limits: {{}}
                    env: {{}}
                    '''
                ).format(command_to_execute=command_to_execute)
            )

        with open('yasmcore/bin/check', 'w') as check:
            check.write(
                textwrap.dedent(
                    '''
                    # coding: utf-8

                    import os
                    import sys
                    import errno


                    pidfile = sys.argv[1]
                    if not os.path.exists(pidfile):
                        sys.exit(1)

                    pid = int(next(iter(open(pidfile))).strip())
                    try:
                        os.kill(pid, 0)
                    except EnvironmentError as e:
                        if e.errno in (errno.ESRCH, errno.EPERM):
                            raise
                    '''))

        with open('yasmcore/bin/stop', 'w') as stop:
            stop.write(
                textwrap.dedent(
                    '''
                    # coding: utf-8

                    import os
                    import sys
                    import signal


                    port = sys.argv[1]
                    os.system('/usr/bin/curl -s --connect-timeout 10 --max-time 20 http://localhost:%s/shutdown/' % (port,))

                    pidfile = sys.argv[2]
                    if os.path.exists(pidfile):
                        pid = int(next(iter(open(pidfile))).strip())
                        os.kill(pid, signal.SIGINT)
                        os.unlink(pidfile)
                    '''
                )
            )

    def on_execute(self):
        self.checkout()
        self.write_scripts()

        run_process(['tar', '-C', 'yasmcore', '-czvf', 'skycore-yasm-agent.tgz', '.'], log_prefix='bundle')

        self.create_resource(
            'skycore-yasm-agent (%s)' % (self.descr, ),
            self.path('skycore-yasm-agent.tgz'),
            resource_types.YASMAGENT_SKYCORE,
            arch=self.arch,
        )


__Task__ = BuildSkycoreYasmAgent
