"""Build YASM DOCS."""

import os
import logging
import tarfile

from sandbox import common
import sandbox.common.types.client as ctc

from sandbox.projects.yasm import resource_types
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk import ssh

from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.sandboxapi import RELEASE_STABLE

# do not delete - needed for environments dynamic creation
from sandbox.projects.common.environments import *  # noqa
from sandbox.sandboxsdk.environments import *  # noqa


class ResourceDescriptionParameter(SandboxStringParameter):
    name = 'description'
    description = 'Resource description'
    default_value = 'YASM docs in .tar.bz2 file'


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git tag'
    default_value = 'master'
    required = False


class BuildYasmDocs(nanny.ReleaseToNannyTask, SandboxTask):
    """Build YASM docs."""

    type = 'BUILD_YASM_DOCS'
    GIT_URL = 'ssh://git@bb.yandex-team.ru/search_infra/yasm.git'
    VAULT_OWNER = "YASM"
    SSH_PRIVATE_KEY_VAULT_NAME = "robot-yasm-golovan-ssh-private-key"
    TGZ_NAME = 'yasm-docs.tar.bz2'
    VENV_NAME = 'python'
    BUILD_DIR = 'build_dir'

    input_parameters = [ResourceDescriptionParameter, GitTagParameter]
    execution_space = 1000
    client_tags = ctc.Tag.LINUX_PRECISE

    def on_execute(self):
        import virtualenv

        tag_name = self.ctx.get('tag') or 'master'
        self.ctx['tag_name'] = tag_name

        logging.info('Creating virtualenv...')
        common.fs.make_folder(self.path(self.BUILD_DIR))
        virtualenv_path = os.path.join(self.path(self.BUILD_DIR), self.VENV_NAME)
        virtualenv.create_environment(virtualenv_path)
        logging.info('Installing requirements in to the virtualenv...')
        pip_path = os.path.join(virtualenv_path, 'bin', 'pip')
        run_process(
            [
                pip_path,
                'install',
                '-i', 'https://pypi.yandex-team.ru/simple/',
                'sphinx'
            ],
            log_prefix='install_sphinx',
            work_dir=self.BUILD_DIR
        )

        logging.info('Building YASM docs from %s tag...', tag_name)
        sources_folder = self.checkout_sources(tag_name)
        yasmapi = os.path.join(sources_folder, 'yasmapi')
        os.chdir(yasmapi)
        docs_path = os.path.join(sources_folder, 'docs')
        res_path = os.path.join(docs_path, '_build')
        sphinx_path = os.path.join(virtualenv_path, 'bin', 'sphinx-build')
        run_process([sphinx_path, '-b', 'html', 'docs', '../docs/_build'],
                    log_prefix='build_docs',
                    work_dir='.')

        if not os.path.exists(res_path):
            raise SandboxTaskFailureError('Docs folder {} does not exist.'.format(res_path))

        logging.info('Creating %s...', self.TGZ_NAME)
        resource_path = self.path(self.TGZ_NAME)
        with tarfile.open(resource_path, 'w:bz2') as tar:
            for entry in os.listdir(res_path):
                tar.add(os.path.join(res_path, entry), entry)

        self.create_resource(
            description=self.ctx.get('description', 'YASM DOCS, tag {}'.format(tag_name)),
            resource_path=resource_path,
            resource_type=resource_types.YASM_DOCS,
            arch='linux'
        )

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        tag_name = self.ctx['tag_name']

        if additional_parameters['release_status'] == RELEASE_STABLE:
            service_id = "production_yasm_nginx_oldgui"
        else:
            return

        nanny_oauth_token = self.get_vault_data('YASM', 'yasm_robot_oauth_token')
        nanny_client = nanny.NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=nanny_oauth_token)

        logging.info('Scheduling deploy of YASM docs...')
        comment = 'Upgrade to {}'.format(tag_name)

        nanny_client.update_service_sandbox_resources(
            service_id=service_id, task_type=self.type, task_id=str(self.id),
            comment=comment, deploy=True, deploy_comment=comment)

        logging.info('YASM docs has been successfully scheduled for deploy.')

    def checkout_sources(self, tag):
        logging.info('Checking out %s tag of YASM repo...', tag)
        src_dir = self.path('yasm-ui')
        with ssh.Key(self, self.VAULT_OWNER, self.SSH_PRIVATE_KEY_VAULT_NAME):
            run_process(['git', 'clone', self.GIT_URL, src_dir], log_prefix='git_clone')
            run_process(['git', 'checkout', tag], work_dir=src_dir, log_prefix='git_checkout')
        return src_dir


__Task__ = BuildYasmDocs
