import os
import tarfile
import sandbox.common.types.misc as ctm

from sandbox.projects.common.nanny import nanny
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.path import Path

from sandbox.sandboxsdk import environments


class YasmStorybook(sdk2.Resource):
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = ['INFRACLOUDUI']


class InfracloudComponentsStorybook(sdk2.Resource):
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = ['INFRACLOUDUI']


class JugglerStorybook(sdk2.Resource):
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = ['INFRACLOUDUI']


class JugglerUI(sdk2.Resource):
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = ['INFRACLOUDUI']


class JugglerUITests(sdk2.Resource):
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = ['INFRACLOUDUI']


class YasmSnapchart(sdk2.Resource):
    releasable = True
    any_arch = False
    executable = False
    auto_backup = True
    releasers = ['YASM']


class YasmUi(sdk2.Resource):
    releasable = True
    any_arch = False
    executable = False
    auto_backup = True
    releasers = ['YASM']


class BuildYasmUi(nanny.ReleaseToNannyTask2, sdk2.Task):
    work_dir = 'yasm'
    snapchart_tarname = 'yasm-snapchart.tgz'
    ui_tarname = 'yasm-ui.tar.bz2'

    class Requirements(sdk2.Requirements):
        environments = (
            environments.NodeJS('10.14.2'),
            environments.GCCEnvironment(version="5.3.0"),
        )

        disk_space = 1024 * 5
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Git parameters") as git_block:
            repository = sdk2.parameters.String("Repository", required=True, default="ssh://git@bb.yandex-team.ru/search_infra/yasm.git")
            tag = sdk2.parameters.String("Branch", default="master")
            commit = sdk2.parameters.String("Commit", default=None)

        with sdk2.parameters.Group("Vault") as vault_block:
            ssh_vault_name = sdk2.parameters.String("Vault item with ssh key for git access", default="robot-yasm-golovan-ssh-private-key")
            ssh_vault_owner = sdk2.parameters.String("Vault item owner", default="YASM")

    def git_clone(self):
        with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            git = Git(self.Parameters.repository)
            git.clone(self.work_dir, self.Parameters.tag, self.Parameters.commit)

    def build_snapchart(self):
        snapchart_path = str(self.path(self.work_dir)) + '/snapchart'

        with sdk2.helpers.ProcessLog(self, logger='build_snapchart') as pl:
            sp.check_call(
                ['make'],
                cwd=snapchart_path,
                stdout=pl.stdout,
                stderr=pl.stderr
            )

        with tarfile.open(self.snapchart_tarname, 'w:gz') as tar:
            os.chdir(snapchart_path)

            tar.add('build/main.js', 'main.js')
            tar.add('node_modules/canvas-prebuilt')
            tar.close()

        resource_path = self.path(self.snapchart_tarname)

        resource = sdk2.ResourceData(YasmSnapchart(
            self,
            'YASM SNAPCHART, tag {}'.format(self.Parameters.tag),
            self.snapchart_tarname
        ))

        resource.path.write_bytes(Path(resource_path).read_bytes())

    def build_ui(self):
        ui_path = str(self.path(self.work_dir)) + '/ui'
        dist_path = os.path.join(ui_path, 'dist')
        tar_path = self.path(self.ui_tarname)
        tag = self.Parameters.tag

        if tag.startswith('release/'):
            os.environ['CI'] = 'true'
            os.environ['SENTRY_AUTH_TOKEN'] = sdk2.Vault.data('INFRACLOUDUI', 'env.SENTRY_AUTH_TOKEN')

        with sdk2.helpers.ProcessLog(self, logger='build_ui') as pl:
            sp.check_call(
                ['make'],
                cwd=ui_path,
                stdout=pl.stdout,
                stderr=pl.stderr
            )

        with tarfile.open(self.ui_tarname, 'w:bz2') as tar:
            for entry in os.listdir(dist_path):
                tar.add(os.path.join(dist_path, entry), entry)

        resource = sdk2.ResourceData(YasmUi(
            self,
            'YASM UI, tag {}'.format(self.Parameters.tag),
            self.ui_tarname
        ))

        resource.path.write_bytes(Path(tar_path).read_bytes())

    def on_execute(self):
        self.git_clone()
        self.build_ui()
        self.build_snapchart()
