import logging

from sandbox import sdk2
from sandbox import common

import requests
import json
import time
import datetime
from enum import Enum
from collections import defaultdict


def get_start_of_day():
    dt = datetime.date.today()
    return time.mktime(dt.timetuple())


def get_start_of_week():
    dt = datetime.date.today()
    day = dt.day - dt.weekday()
    if day <= 0:
        day = 1
    first_day = datetime.date(dt.year, dt.month, day)
    return time.mktime(first_day.timetuple())


def get_start_of_month():
    dt = datetime.date.today()
    first_day = datetime.date(dt.year, dt.month, 1)
    return time.mktime(first_day.timetuple())


class Since(Enum):
    TODAY = 1  # since 00:00
    WEEK = 2  # since 00:00 last monday
    MONTH = 3  # since 00:00 first day of month
    LAST_HOUR = 4  # last hour
    LAST_DAY = 5  # last 24 hours
    LAST_WEEK = 6  # last 7 days
    LAST_MONTH = 7  # last 30 days
    PREV_DAY = 8


class SinceFormer:

    funcs = {
        Since.TODAY: get_start_of_day(),
        Since.WEEK: get_start_of_week(),
        Since.MONTH: get_start_of_month(),
        Since.LAST_HOUR: time.time() - 60 * 60,
        Since.LAST_DAY: time.time() - 60 * 60 * 24,
        Since.LAST_WEEK: time.time() - 60 * 60 * 24 * 7,
        Since.LAST_MONTH: time.time() - 60 * 60 * 24 * 30,
        Since.PREV_DAY: get_start_of_day() - 60 * 60 * 24
    }

    def __init__(self, _val):
        self.since = _val

    def get_since(self, _since=None):
        if not _since:
            return self.funcs[self.since]
        else:
            return self.funcs[_since]


def test_since():
    since_obj = SinceFormer(Since.TODAY)
    print 'current time', (time.time()), datetime.datetime.fromtimestamp(time.time())
    print since_obj.get_since(), datetime.datetime.fromtimestamp(since_obj.get_since())
    print since_obj.get_since(Since.WEEK), datetime.datetime.fromtimestamp(since_obj.get_since(Since.WEEK))
    print since_obj.get_since(Since.MONTH), datetime.datetime.fromtimestamp(since_obj.get_since(Since.MONTH))

    print since_obj.get_since(Since.LAST_HOUR), datetime.datetime.fromtimestamp(since_obj.get_since(Since.LAST_HOUR))
    print since_obj.get_since(Since.LAST_DAY), datetime.datetime.fromtimestamp(since_obj.get_since(Since.LAST_DAY))
    print since_obj.get_since(Since.LAST_WEEK), datetime.datetime.fromtimestamp(since_obj.get_since(Since.LAST_WEEK))
    print since_obj.get_since(Since.LAST_MONTH), datetime.datetime.fromtimestamp(since_obj.get_since(Since.LAST_MONTH))


def get_yasm_checks():
    r = requests.get('http://juggler.search.yandex.net:8998/api/checks/checks?do=1&tag_name=a_prj_yasm')
    if r.status_code == 200:
        response = r.json()
        check_pairs = []
        for host, checks in response.iteritems():
            check_pairs += [[host, check] for check in checks.keys()]
        return check_pairs
    return []


def get_checks_history(_host, _service, _since, _until=None):
    page_size = 50
    headers = {"content-type": "application/json"}
    data = {
        "host": _host,
        "service": _service,
        "page_size": page_size,
        "statuses": ["CRIT"],
        "since": _since
    }
    if not _until:
        data["until"] = time.time()
    else:
        data["until"] = _until

    crits = []
    backend = ''
    cur_page = 1
    while True:
        data["page"] = cur_page
        r = requests.post('http://juggler-api.search.yandex.net/v2/history/get_check_history',
                          headers=headers,
                          data=json.dumps(data))

        if r.status_code == 200:
            response = r.json()
            backend = response.get('meta', {}).get('_backend', 'unknown')
            items = response.get('states', {})
            cur_page += 1
            for item in items:
                item_status = item['status']
                if item_status == 'OK' or item_status == 'WARN':
                    continue

                crit_time = item['start_time']
                crit_description = item['description']
                dt = datetime.datetime.fromtimestamp(crit_time)
                crit = {'time': dt, 'description': crit_description}
                crits.append(crit)
            if len(items) < page_size:
                break
    return crits, backend


def analyze_checks_history(_since=Since.LAST_DAY, _until=None, _token=''):
    since_obj = SinceFormer(_since)
    checks = get_yasm_checks()

    logging.info('Getting CRITS from {} to {}'.format(
        datetime.datetime.fromtimestamp(since_obj.get_since(_since)),
        datetime.datetime.fromtimestamp(_until)))
    total_crits = 0
    crit_dict = {}
    groups = defaultdict(list)
    values = []

    for check in checks:
        crits, backend = get_checks_history(check[0], check[1], since_obj.get_since(_since), _until)

        flapping = len([x for x in crits if 'flapping' in x['description']])

        if len(crits) > 0:
            crit_dict['{}:{}'.format(check[0], check[1])] = [len(crits), flapping]
            groups[check[1]].append([check[0], len(crits), flapping])
            logging.info(
                '{};\\t{}\\t;{};{}'.format(
                    datetime.date.fromtimestamp(since_obj.get_since(_since)),
                    '\\t'.join(check),
                    len(crits),
                    flapping
                )
            )
            values.append({'fielddate': '{}'.format(datetime.date.fromtimestamp(since_obj.get_since(_since))),
                           'check': '\t{}\t'.format('\t'.join(check)),
                           'count': len(crits),
                           'flaps': flapping})
    print total_crits
    send_data_to_stat(values, _token)


def send_data_to_stat(_values, _token):
    headers = {"Authorization": "OAuth %s" % _token}
    r = requests.post(
        'https://upload.stat.yandex-team.ru/_api/report/data',
        headers=headers,
        data={
            "name": 'YASM/test',
            "scale": 'd',
            "data": json.dumps({'values': _values})
        }
    )
    resp = r.json()
    logging.info(r.status_code, resp.get('message', 'no message'))


class RunYasmStatTask(sdk2.Task):
    """Run YASM Statistics Task"""

    releasers = ['pirogov']

    def on_execute(self):
        secret_name = "yasm_stat_token"
        try:
            stat_token = sdk2.Vault.data(secret_name)
            # logging.info(uniquely_named_secret)

            analyze_checks_history(Since.PREV_DAY, get_start_of_day(), stat_token)
        except common.errors.VaultError:
            raise common.errors.TaskFailure("Token not found!")
