from sandbox import sdk2
from sandbox import common

import requests
import datetime
from random import randrange
import json
import logging


def median(_list):
    sorts = sorted(_list)
    length = len(sorts)
    if not length % 2:
        return (sorts[length / 2] + sorts[length / 2 - 1]) / 2.0
    return sorts[length / 2]


class YasmStatsCalculator():
    def __init__(self, port=12901, log=None, user=None, pwd=None, token=None):
        self.log = log
        self.port = port
        self.user = user
        self.pwd = pwd
        self.stat_token = token
        self.collectors = []
        self.collector = ''
        self.collector_url = ''
        self.yasm_url = 'https://yasm.yandex-team.ru'

        self.collectors = self.get_collectors_list()
        self.collector = self.get_random_collector()
        if len(self.collector) > 0:
            self.collector_url = 'http://{}:{}/?user={}&password={}'.format(self.collector, self.port, self.user, self.pwd)
        logging.info(self.collector)

    def get_collectors_list(self):
        r = requests.get('{}/metainfo/hosts/?itype=yasmcollector&ctype=prod&prj=yasm'.format(self.yasm_url))
        if r.status_code != 200:
            return []
        resp = r.json()
        status = resp.get('status', 'error')
        if status != 'ok':
            return []
        response = resp.get('response', {})
        collector_list = [host['name'] for host in response.get('result', [])]
        return collector_list

    def get_random_collector(self):
        count = len(self.collectors)
        collector = ''
        if count > 0:
            ind = randrange(count)
            collector = self.collectors[ind]
        return collector

    def get_table_version(self):
        if len(self.collector_url) == 0:
            return 0
        version = 0
        query = "select version from rt_versions order by version desc limit 1"
        r = requests.post(self.collector_url, data=query)
        if r.status_code != 200:
            logging.error('error getting version for instances table - {}'.r.text)
            return 0
        version = int(r.text)
        return version

    def list_itypes(self):
        version = self.get_table_version()
        if version == 0:
            return []
        query = 'select distinct itype from rt_instances_{}'.format(version)
        r = requests.post(self.collector_url, data=query)
        if r.status_code != 200:
            logging.error('error getting itypes - {}'.format(r.text))
            return []
        return sorted(r.text.split())

    def get_signals_stats(self, itype):
        version = self.get_table_version()
        if version == 0:
            return []
        #
        query = "select host, max(sig_count), avg(sig_count), sum(sig_count) from " \
                "(select host, instance_id from rt_instances_{} where itype='{}' and instance_id != 0) " \
                "any inner join (select instance_id, count()-1 as sig_count from rt_signals_{} " \
                "where instance_id !=0 " \
                "and signal not in (SELECT signal from rt_is_common_{}) " \
                "and signal not like '%portoinst-%'" \
                "group by instance_id) using instance_id group by host order by host".format(version, itype, version, version)

        r = requests.post(self.collector_url, data=query)
        if r.status_code != 200:
            logging.error('error getting signals for {} - {}'.format(itype, r.text))
            return []
        stats = [host_stat.split('\t') for host_stat in r.text.split('\n') if len(host_stat)]

        return stats

    def get_signals_count(self, itype):
        version = self.get_table_version()
        if version == 0:
            return 0

        query = "select count() from (SELECT distinct signal from rt_signals_{} " \
                "where instance_id in " \
                "(select instance_id from rt_instances_{} where itype='{}') " \
                "and signal not in (SELECT signal from rt_is_common_{}) " \
                "and signal not like 'portoinst-%' and signal not like 'counter-%')".format(version, version, itype, version)

        r = requests.post(self.collector_url, data=query)
        if r.status_code != 200:
            logging.error('error getting signals count for {} - {}'.format(itype, r.text))
            return 0
        res = 0
        try:
            res = int(r.text)
        except:
            res = 0
        return res

    def calc_yasm_stats(self, itypes=None):
        if itypes is None:
            itypes = self.list_itypes()
        skip_list = []

        stat_values = []
        logging.info('start - {}'.format(datetime.datetime.now()))
        for itype in sorted(itypes):
            if itype in skip_list:
                continue
            stats = self.get_signals_stats(itype)
            max_signals_count = []
            signals_count = []
            sum_signals = 0

            for stat in stats:
                max_signals_count.append(float(stat[1]))
                signals_count.append(float(stat[2]))
                sum_signals += int(stat[3])
            avg_max_signals = 0
            max_max_signals = 0
            min_max_signals = 0
            med_max_signals = 0
            if len(max_signals_count) > 0:
                avg_max_signals = int(sum(max_signals_count) / len(max_signals_count))
                max_max_signals = int(max(max_signals_count))
                min_max_signals = int(min(max_signals_count))
                med_max_signals = int(median(max_signals_count))

            unique_signals = self.get_signals_count(itype)

            logging.info(
                '{}; {}; {}; {}; {}; {}; {}; {}; {}'.format(
                    datetime.date.today(), itype, len(stats), avg_max_signals,
                    med_max_signals, min_max_signals, max_max_signals,
                    sum_signals, unique_signals
                )
            )

            stat_values.append({
                'fielddate': '{}'.format(datetime.date.today()),
                'name': itype,
                'hosts': len(stats),
                'avg': avg_max_signals,
                'med': med_max_signals,
                'min': min_max_signals,
                'max': max_max_signals,
                'sum': sum_signals,
                'uniq': unique_signals
            })

        logging.info('end - {}'.format(datetime.datetime.now()))
        if self.stat_token is not None:
            self.send_data_to_stat(stat_values, self.stat_token)

    def send_data_to_stat(self, _values, _token):
        headers = {"Authorization": "OAuth %s" % _token}
        r = requests.post('https://upload.stat.yandex-team.ru/_api/report/data',
                          headers=headers,
                          data={"name": 'YASM/signals_stats',
                                "scale": 'd',
                                "data": json.dumps({'values': _values})
                                })
        if r.status_code != 200:
            logging.error('error {} posting data to stat - {}'.format(r.status_code, r.text))
            return
        logging.info('data successfully sent to Stat')


class YasmSignalsStatTask(sdk2.Task):
    """Export YASM Signals Statistics To Stat"""

    releasers = ['guest', 'pirogov']

    def on_execute(self):
        vault_token = "yasm_stat_token"
        vault_yasmcollector_user = "yasm_collector_user"
        vault_yasmcollector_pwd = "yasm_collector_pwd"
        try:
            stat_token = sdk2.Vault.data(vault_token)
            yasmcollector_user = sdk2.Vault.data(vault_yasmcollector_user)
            yasmcollector_pwd = sdk2.Vault.data(vault_yasmcollector_pwd)
            logging.info(stat_token, yasmcollector_user, yasmcollector_pwd)

            calc = YasmStatsCalculator(user=yasmcollector_user, pwd=yasmcollector_pwd, token=stat_token)
            calc.calc_yasm_stats(itypes=None)
        except common.errors.VaultError:
            raise common.errors.TaskFailure("Secret not found!")
