# coding: utf-8
import os
import logging
import time
import platform
import shutil
import json
import urllib2
from contextlib import contextmanager

import requests

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.common import errors
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.environments import GCCEnvironment
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter, SandboxRadioParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.paths import make_folder

from sandbox.projects.resource_types import OTHER_RESOURCE
from sandbox.projects.common.environments import SandboxPythonDevEnvironment, SandboxLibSnappyEnvironment
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.juggler import virtualenv, debian
from sandbox.projects.yasm.resource_types import YASMAGENT_DEB, YASMAGENT_VENV
import sandbox.projects.common.gnupg
import sandbox.projects.common.debpkg


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git branch/tag'
    default_value = ''
    required = True


class PublishPackageParameter(SandboxBoolParameter):
    name = 'publish_package'
    description = 'Upload package to dist'
    default_value = False


class StartrekTicketParameter(SandboxStringParameter):
    name = 'startrek_ticket'
    description = 'Startrek ticket'
    default_value = ''


class ReleaseStatusParameter(SandboxRadioParameter):
    name = 'release_status'
    choices = [(x, x) for x in ('none', 'prestable', 'testing', 'unstable')]
    description = 'Release status'
    default_value = 'none'


class RunTestsParameter(SandboxBoolParameter):
    name = 'run_tests'
    description = 'Test build'
    default_value = False


class BaseBuildCoreYasmTask(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BASE_BUILD_CORE_GOLOVAN_TASK'
    dns = ctm.DnsType.DNS64

    environment = [GCCEnvironment(), SandboxPythonDevEnvironment(), SandboxLibSnappyEnvironment()]

    input_parameters = [
        GitTagParameter,
        PublishPackageParameter,
        StartrekTicketParameter,
        ReleaseStatusParameter,
        RunTestsParameter
    ]

    client_tags = ctc.Tag.LINUX_PRECISE
    execution_space = 6000

    SERVICE_NAME = 'redefine_me'  # e.g. 'yasmserver'
    TGZ_PATH = 'redefine_me'  # e.g. '{}.tar.gz'.format(SERVICE_NAME)
    RESOURCE_TYPE = OTHER_RESOURCE
    COPY_DIRS = {}
    GIT_URL = 'ssh://git@bb.yandex-team.ru/search_infra/yasm.git'
    CHECKOUT_PATH = 'yasm'
    REMOVE_DIRS = [CHECKOUT_PATH]
    BUILD_DIR = 'build_dir'
    PLATFORM_INDEPENDENT = False

    VAULT_OWNER = "YASM"
    SSH_PRIVATE_KEY_VAULT_NAME = "robot-yasm-golovan-ssh-private-key"
    GPG_PUBLIC_KEY_VAULT_NAME = "robot-yasm-golovan-gpg-public"
    GPG_PRIVATE_KEY_VAULT_NAME = "robot-yasm-golovan-gpg-private"
    OAUTH_TOKEN_PRIVATE_KEY_VAULT_NAME = 'robot-yasm-golovan-st-oauth'

    YASM_CONF_NAME = "yasm.conf"

    ROBOT_NAME = "robot-yasm-golovan"
    ROBOT_EMAIL = "{}@yandex-team.ru".format(ROBOT_NAME)

    DUPLOAD_CONF = {
        'search': {
            'fqdn': "search.dupload.dist.yandex.ru",
            'method': "scpb",
            'login': ROBOT_NAME,
            'incoming': "/repo/search/mini-dinstall/incoming/",
            'dinstall_runs': 0,
        }
    }

    def _checkout(self):
        logging.info('Checkout code...')

        if 'ref_id' in self.ctx and 'ref_sha' in self.ctx:
            self.ctx[GitTagParameter.name] = self.ctx['ref_id']
            tag = self.ctx['ref_sha']
        else:
            tag = self.ctx['ref_id'] = self.ctx[GitTagParameter.name]

        with ssh.Key(self, self.VAULT_OWNER, self.SSH_PRIVATE_KEY_VAULT_NAME):
            run_process(
                ['git', 'clone', self.GIT_URL, self.CHECKOUT_PATH],
                log_prefix='git_clone',
                shell=True,
            )
            run_process(
                ['git', 'checkout', tag],
                work_dir=self.CHECKOUT_PATH,
                log_prefix='git_checkout',
                shell=True)

    def _assemble_bundle(self, copy_dirs, remove_dirs, yasmconf_path):
        logging.info('Assemble bundle...')
        copy_path(os.path.join(self.CHECKOUT_PATH, 'yasmutil'), os.path.join(self.BUILD_DIR, 'yasmutil'))
        copy_path(os.path.join(self.CHECKOUT_PATH, 'yasmagent'), os.path.join(self.BUILD_DIR, 'yasmagent'))

        shutil.copy(yasmconf_path, os.path.join(self.BUILD_DIR, "yasmagent", self.YASM_CONF_NAME))

        with open(os.path.join(self.BUILD_DIR, 'yasmagent', "VERSION"), "w") as f:
            f.write("yandex-yasmagent (branch {}, task id {})".format(self.ctx['tag'], self.id))

        for copy_dir in copy_dirs.keys():
            copy_path(os.path.join(self.CHECKOUT_PATH, copy_dir), os.path.join(self.BUILD_DIR, copy_dirs[copy_dir]))
        for remove_dir in remove_dirs:
            remove_path(remove_dir)

    def _make_resource(self, path, version, tarball_path, resource_type, service_name):
        logging.info('Creating tgz file...')
        run_process(['tar', '-czf', self.path(tarball_path)] + list(os.listdir(path)), work_dir=path)
        if self.PLATFORM_INDEPENDENT:
            attributes = {'version': version}
            arch = 'any'
        else:
            attributes = {'version': version, 'platform': platform.platform()}
            arch = 'linux'
        self.create_resource(
            description='{} tarball version {} from branch/tag {}'.format(service_name, version, self.ctx['tag']),
            resource_path=tarball_path,
            resource_type=resource_type,
            attributes=attributes,
            arch=arch,
        )

    def _prepare_core_virtualenv(self, venv_path, target_prefix=None):
        root_path = self.path(self.CHECKOUT_PATH)
        requirements_list = [
            os.path.join(root_path, 'requirements.txt'),
            os.path.join(root_path, 'requirements-arcadia.yaml')
        ]
        with virtualenv.venv_context(self, self.path('tmp_core_venv'), requirements_list,
                                     target_prefix=target_prefix, python_path='/skynet/python',
                                     root_path=root_path) as venv:

            self._generate_groups_cache(venv.executable)
            yasmconf_path = self._generate_yasmconf(venv.executable)

            venv.make_relocatable()
            shutil.copytree(venv.root_dir, venv_path)

        return yasmconf_path

    def _prepare_agent_virtualenv(self, venv_path, target_prefix=None, yasmconf_path=None):
        root_path = self.path(self.CHECKOUT_PATH)
        requirements_list = [
            os.path.join(root_path, 'requirements-agent.txt'),
            os.path.join(root_path, 'requirements-agent-arcadia.yaml')
        ]
        with virtualenv.venv_context(self, self.path('tmp_agent_venv'), requirements_list,
                                     target_prefix=target_prefix, python_path='/skynet/python',
                                     root_path=root_path) as venv:
            shutil.copytree(venv.root_dir, venv_path)
            if yasmconf_path is not None:
                shutil.copy(yasmconf_path, os.path.join(venv_path, self.YASM_CONF_NAME))

    def _generate_yasmconf(self, executable):
        generator_path = os.path.join(self.path(self.CHECKOUT_PATH), "scripts", "conf_scripts", "makeyasmconf2.py")

        yasmconf_path = os.path.join(self.path(self.BUILD_DIR), self.YASM_CONF_NAME)
        run_process([executable, generator_path,
                     '--yasmagent-dir=%s' % self.path(self.CHECKOUT_PATH),
                     '--yasmutil-dir=%s' % self.path(self.CHECKOUT_PATH),
                     '--conf-dir=%s' % os.path.join(self.path(self.CHECKOUT_PATH), 'CONF'),
                     '--out=%s' % yasmconf_path,
                     '--log-level=INFO'],
                    log_prefix='build_config',
                    work_dir=self.CHECKOUT_PATH)
        return yasmconf_path

    def _generate_groups_cache(self, executable):
        cache_dir = os.path.join(self.path(self.CHECKOUT_PATH), 'groups_cached')
        conf_scripts = os.path.join(self.path(self.CHECKOUT_PATH), 'scripts/conf_scripts')
        make_folder(cache_dir)
        run_process(
            [executable, './make_host_lists.py', '--out-dir', cache_dir],
            log_prefix='generate_groups_cache',
            work_dir=conf_scripts
        )

    def _download_latest_resource(self, parameter_name, resource_type):
        resource_id = self.ctx.get(parameter_name)
        if resource_id:
            resource = channel.channel.sandbox.get_resource(resource_id)
        else:
            resource_list = channel.channel.sandbox.list_resources(
                order_by="-id", limit=1, status="READY",
                resource_type=resource_type,
                all_attrs={'released': 'stable'}
            )
            if not resource_list:
                raise errors.TaskFailure('No module found!')
            else:
                resource = resource_list[0]
        return self.sync_resource(resource)

    def _check_agent_versions(self, root_dir):
        package_version = debian.extract_changelog_version(root_dir)
        if not self.ctx.get(PublishPackageParameter.name):
            return package_version

        reply = None
        for retry in xrange(5):
            try:
                reply = json.load(urllib2.urlopen('http://dist.yandex.ru/api/v1/search?pkg=yandex-yasmagent&strict=true'))
            except Exception as exc:
                logging.exception('request to dist failed: %s', exc)
                time.sleep(1)
            else:
                break

        if reply is None or not reply['success']:
            raise errors.TaskFailure('No reply from dist!')
        elif package_version in {row['version'] for row in reply['result']}:
            raise errors.TaskFailure('Agent package {} already exists in dist!'.format(package_version))

        return package_version

    def _build_and_publish_agent(self, root_dir, yasmconf_path):
        package_version = self._check_agent_versions(root_dir)
        with open(os.path.join(root_dir, 'VERSION'), 'w') as stream:
            stream.write("yandex-yasmagent ({})\n".format(package_version))

        shutil.copy(yasmconf_path, os.path.join(root_dir, self.YASM_CONF_NAME))

        with sandbox.projects.common.gnupg.GpgKey(self, self.VAULT_OWNER, self.GPG_PRIVATE_KEY_VAULT_NAME, self.GPG_PUBLIC_KEY_VAULT_NAME):
            run_process('debuild --no-tgz-check -k{}'.format(self.ROBOT_EMAIL),
                        work_dir=root_dir, wait=True, check=True, log_prefix='debuild')

            if self.ctx.get(PublishPackageParameter.name):
                with sandbox.projects.common.debpkg.DebRelease(self.DUPLOAD_CONF) as deb:
                    with ssh.Key(self, self.VAULT_OWNER, self.SSH_PRIVATE_KEY_VAULT_NAME):
                        deb.debrelease(['--to', 'search'], work_dir=root_dir)

                logging.info('Yasmagent has been successfully loaded to dist.')

        self.create_resource(
            arch='linux',
            resource_path=os.path.join(root_dir, '..', 'yandex-yasmagent_{}_all.deb'.format(package_version)),
            resource_type=YASMAGENT_DEB,
            description='Yasmagent deb {}'.format(package_version),
        )

    def _comment_startrek_ticket(self):
        startrek_ticket = self.ctx.get(StartrekTicketParameter.name)

        if not startrek_ticket:
            return

        url = 'https://st-api.yandex-team.ru/v2/issues/{}/comments'.format(startrek_ticket)

        oauth_token = self.get_vault_data(self.VAULT_OWNER, self.OAUTH_TOKEN_PRIVATE_KEY_VAULT_NAME)
        headers = {
            'Content-Type': 'application/json',
            'Authorization': 'OAuth {}'.format(oauth_token),
        }

        data = {
            'text': 'Finished **{0}**: ((https://sandbox.yandex-team.ru/task/{1} {1}))'.format(
                self.type,
                self.id,
            )
        }

        requests.post(url, json=data, headers=headers)

    def _release_hook(self):
        release_status = self.ctx.get(ReleaseStatusParameter.name)

        if release_status and release_status != 'none':
            releaser = self.create_subtask(
                task_type='RELEASE_ANY',
                input_parameters={
                    'release_task_id': self.id,
                    'release_status': release_status,
                },
                description='BUILD_YASM_CORE ({}) autorelease'.format(self.id),
            )

            self.info = "Subtask {} started, waiting for it's decision about release.\n\n".format(
                releaser.id
            )

    def _pack_test_logs(self, category, test_report_path):
        if os.path.exists(test_report_path):
            self.create_resource(
                description="{} test HTML report".format(category),
                resource_path=test_report_path,
                resource_type=OTHER_RESOURCE
            )

    def _run_py_test(self, category, venv, *args):
        test_report_path = os.path.join(self.path(self.BUILD_DIR), "{}_test_report.html".format(category))
        run_tests_cmd = [
            venv.executable,
            os.path.join(venv.root_dir, "bin", "py.test"),
            "--html={}".format(test_report_path),
            "--self-contained-html"
        ]
        run_tests_cmd.extend(args)
        try:
            run_process(run_tests_cmd, log_prefix="pytest_{}".format(category), work_dir=self.path(self.CHECKOUT_PATH))
        finally:
            self._pack_test_logs(category, test_report_path)

    @contextmanager
    def _core_testing_venv(self):
        checkout_path = self.path(self.CHECKOUT_PATH)
        requirements_list = [
            os.path.join(checkout_path, 'requirements.txt'),
            os.path.join(checkout_path, 'requirements-arcadia.yaml'),
            os.path.join(checkout_path, 'test', 'requirements.txt')
        ]
        with virtualenv.venv_context(self, self.path('test_core_venv'), requirements_list, root_path=checkout_path) as venv:
            yield venv

    @contextmanager
    def _agent_testing_venv(self):
        checkout_path = self.path(self.CHECKOUT_PATH)
        requirements_list = [
            os.path.join(checkout_path, 'requirements-agent.txt'),
            os.path.join(checkout_path, 'requirements-agent-arcadia.yaml'),
            os.path.join(checkout_path, 'test', 'requirements-agent.txt')
        ]
        with virtualenv.venv_context(self, self.path('test_agent_venv'), requirements_list, root_path=checkout_path) as venv:
            yield venv

    def _run_tests(self, build_dir):
        checkout_path = self.path(self.CHECKOUT_PATH)

        with self._core_testing_venv() as venv:
            self._run_py_test("core", venv, "--ignore=test/yasmagent", os.path.join(checkout_path, "test"))

        with self._agent_testing_venv() as venv:
            self._run_py_test("agent", venv, os.path.join(checkout_path, "test", "yasmagent"))

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get('tag'), None

    def on_execute(self):
        """
        Plan is:
        * set CC and CXX to our gcc
        * git clone and checkout specified branch/tag
        * create virtualenv
        * install requirements
        * make virtualenv relocatable
        * assemble bundle with only needed files
        * create tarball with bundle
        """
        version = int(time.time())
        self._checkout()

        build_dir = self.path(self.BUILD_DIR)
        make_folder(build_dir)

        if self.ctx.get(RunTestsParameter.name) or ('ref_id' in self.ctx and 'ref_sha' in self.ctx):
            self.ctx[RunTestsParameter.name] = True
            self._run_tests(build_dir)

        core_python_path = os.path.join(build_dir, 'python')
        yasmconf_path = self._prepare_core_virtualenv(core_python_path)

        agent_python_path = os.path.join(build_dir, 'agent_python')
        self._prepare_agent_virtualenv(agent_python_path, yasmconf_path=yasmconf_path)

        self._make_resource(agent_python_path, version, 'yasmagent_venv.tar.gz', YASMAGENT_VENV, 'yasmagent venv')

        self._assemble_bundle(copy_dirs=self.COPY_DIRS, remove_dirs=self.REMOVE_DIRS,
                                yasmconf_path=yasmconf_path)
        os.rename(agent_python_path, os.path.join(build_dir, 'yasmagent', 'python'))

        self._make_resource(build_dir, version, self.TGZ_PATH, self.RESOURCE_TYPE, self.SERVICE_NAME)

        self._build_and_publish_agent(os.path.join(build_dir, 'yasmagent'), yasmconf_path)

        self._release_hook()

        self._comment_startrek_ticket()
