import datetime

from sandbox import sdk2
from sandbox.projects.ydo import execute_cmd, ydo_releasers
from sandbox.projects.ydo.parameters_view import generate_view_for_yt


class YdoWorkerStatsAccumulatorBinary(sdk2.Resource):
    releasers = ydo_releasers
    executable = True
    releasable = True


class YdoAccumulateWorkerStats(sdk2.Task):
    """ Accumulate worker stats for YDO. """

    class Parameters(generate_view_for_yt(
        ydo_worker_stats_accumulator=YdoWorkerStatsAccumulatorBinary,
        check_owner=True,
        with_environ=True
    )):
        input_date = sdk2.parameters.RadioGroup(
            'Input date',
            choices=(
                ('yesterday', 'yesterday'),
                ('custom', 'custom'),
            ),
            required=True,
            default='yesterday',
            sub_fields={'custom': ['custom_input_date']}
        )
        custom_input_date = sdk2.parameters.StrictString(
            'YYYY-MM-DD',
            regexp=r'\d{4}-\d{2}-\d{2}',
            required=True
        )

        stat_types = sdk2.parameters.CheckGroup(
            'Stats to calculate',
            choices=(
                ('shows', 'shows'),
                ('clicks', 'clicks'),
                ('contact_clicks', 'contact_clicks'),
                ('serp_to_profile_clicks', 'serp_to_profile_clicks'),
                ('promotion_good_actions', 'promotion_good_actions'),
            ),
            required=True
        )

        stat_days = sdk2.parameters.List(
            'Day intervals to calculate stats',
            required=True
        )

        input_path = sdk2.parameters.String(
            'Input node path',
            required=True,
            default='//home/ydo/users/zmiecer/worker-clicks'
        )

        output_path = sdk2.parameters.String(
            'Output node path',
            required=True,
            default_value='//home/ydo/users/zmiecer/worker-stats'
        )

        expiration_time = sdk2.parameters.Integer(
            'Expiration time of the output table in days',
            required=True,
            default=90
        )

        change_current = sdk2.parameters.Bool('Make "current" link on result table.')

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_input_date(self):
        if self.Parameters.input_date == 'yesterday':
            yesterday = datetime.datetime.today() - datetime.timedelta(days=1)
            input_date = yesterday.strftime('%Y-%m-%d')
        elif self.Parameters.input_date == 'custom':
            input_date = self.Parameters.custom_input_date
        return input_date

    def on_execute(self):
        input_date = self.get_input_date()
        cmd = [
            self.Parameters.ydo_worker_stats_accumulator_path,
            '--input_date', input_date,
            '--input_path', self.Parameters.input_path,
            '--output_path', self.Parameters.output_path,
            '--expiration_time', str(self.Parameters.expiration_time)  # sp.Popen can't work with int
        ]
        cmd.append('--stats')
        cmd.extend(self.Parameters.stat_types)
        cmd.append('--stat_days')
        cmd.extend(self.Parameters.stat_days)
        if self.Parameters.change_current:
            cmd.append('--change_current')

        execute_cmd(
            cmd=cmd,
            log_name='ydo_accumulate_worker_stats',
            error_message='Worker stats accumulator failed',
            env=self.Parameters.get_environ()
        )
