import datetime
import logging
import os
import sandbox.common.types.task as ctt

from sandbox import sdk2

from sandbox.projects.saas.common.resources import SAAS_UTIL_GET_STANDALONE_INDEXER

from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.accumulate_worker_stats import YdoAccumulateWorkerStats
from sandbox.projects.ydo.prepare_saas_stats import YdoPrepareSaasStats
from sandbox.projects.ydo.solomon_mixin import SolomonMixin


class YdoAggregateWorkerStatsAndPublishToSaas(sdk2.Task, SolomonMixin):
    """ Chain task that accumulates statistic and publishes to SaaS. """
    class Context(sdk2.Task.Context):
        current_subtask_id = None

    class Parameters(sdk2.Task.Parameters):
        mode = sdk2.parameters.RadioGroup(
            'Mode',
            choices=(
                ('stable', 'stable'),
                ('testing_or_stable', 'testing_or_stable'),
            ),
            required=True,
            default='stable'
        )

        input_date = sdk2.parameters.RadioGroup(
            'Input date',
            choices=(
                ('yesterday', 'yesterday'),
                ('custom', 'custom'),
            ),
            required=True,
            default='yesterday',
            sub_fields={'custom': ['custom_input_date']}
        )
        custom_input_date = sdk2.parameters.StrictString(
            'YYYY-MM-DD',
            regexp=r'\d{4}-\d{2}-\d{2}',
            required=True
        )

        stat_types = sdk2.parameters.CheckGroup(
            'Stats to calculate',
            choices=(
                ('shows', 'shows'),
                ('clicks', 'clicks'),
                ('contact_clicks', 'contact_clicks'),
                ('serp_to_profile_clicks', 'serp_to_profile_clicks'),
                ('promotion_good_actions', 'promotion_good_actions'),
            ),
            required=True
        )

        stat_days = sdk2.parameters.List(
            'Day intervals to calculate stats. Just enter numbers.',
            required=True
        )

        change_current = sdk2.parameters.Bool('Make "current" link on last worker-stats table.')

        input_path = sdk2.parameters.String(
            'Input node path',
            required=True,
            default='//home/ydo/logs/prod/parsed_sessions'
        )

        temp_node_path = sdk2.parameters.String(
            'Temp node path. It should have directories worker-stats and saas-worker-stats inside.',
            required=True,
            default='//home/ydo/session_parsing/test'
        )

        saas_node_path = sdk2.parameters.String(
            'SaaS node path.',
            required=True,
            default='//home/ydo/users/zmiecer/saas'
        )

        ctype = sdk2.parameters.RadioGroup(
            'ctype',
            choices=(
                ('prestable', 'prestable'),
                ('stable', 'stable'),
                ('stable_patents', 'stable_patents'),
                ('stable_kv', 'stable_kv'),
                ('stable_middle_kv', 'stable_middle_kv'),
            ),
            required=True,
            default='prestable'
        )

        temp_tables_expiration_time = sdk2.parameters.Integer(
            'Temp tables expiration time',
            required=True,
            default=14
        )

        publish = sdk2.parameters.Bool(
            'Whether to publish index',
            default=False
        )

        env = sdk2.parameters.Dict(
            'Environment variables',
            required=True,
            default={'YT_PROXY': 'hahn.yt.yandex.net'}
        )

        secret_env = sdk2.parameters.Dict(
            'Secret environment variables',
            required=True,
            default={'YT_TOKEN': 'yt-token'}
        )

        with sdk2.parameters.RadioGroup('Environment') as solomon_env:
            solomon_env.values['production'] = solomon_env.Value(value='Production')
            solomon_env.values['testing'] = solomon_env.Value(value='Testing', default=True)

    def get_input_date(self):
        if self.Parameters.input_date == 'yesterday':
            yesterday = datetime.datetime.today() - datetime.timedelta(days=1)
            input_date = yesterday.strftime('%Y-%m-%d')
        elif self.Parameters.input_date == 'custom':
            input_date = self.Parameters.custom_input_date
        return input_date

    def accumulate_worker_stats(self):
        logging.info('Starting worker stats accumulation.')
        stats_accumulator = YdoAccumulateWorkerStats(
            self,
            description="Accumulate worker stats for {}".format(self.id),
        )
        stats_accumulator.Parameters.owner = self.owner
        stats_accumulator.Parameters.release_chooser = self.Parameters.mode
        stats_accumulator.Parameters.input_date = self.Parameters.input_date
        stats_accumulator.Parameters.stat_types = self.Parameters.stat_types
        stats_accumulator.Parameters.stat_days = self.Parameters.stat_days
        stats_accumulator.Parameters.expiration_time = self.Parameters.temp_tables_expiration_time
        stats_accumulator.Parameters.input_path = self.Parameters.input_path
        stats_accumulator.Parameters.output_path = '{}/worker-stats'.format(self.Parameters.temp_node_path)
        stats_accumulator.Parameters.env = self.Parameters.env
        stats_accumulator.Parameters.secret_env = self.Parameters.secret_env
        stats_accumulator.Parameters.change_current = self.Parameters.change_current
        stats_accumulator.Parameters.kill_timeout = 7 * 60 * 60  # 7 hours

        stats_accumulator.save().enqueue()
        return stats_accumulator.id

    def prepare_stats_for_saas(self):
        logging.info('Starting SaaS stats preparation.')
        saas_converter = YdoPrepareSaasStats(
            self,
            description="Prepare worker stats for SaaS for {}".format(self.id),
        )

        saas_converter.Parameters.owner = self.owner
        saas_converter.Parameters.release_chooser = self.Parameters.mode
        saas_converter.Parameters.input_date = self.Parameters.input_date
        saas_converter.Parameters.expiration_time = self.Parameters.temp_tables_expiration_time
        saas_converter.Parameters.input_path = '{}/worker-stats'.format(self.Parameters.temp_node_path)
        saas_converter.Parameters.output_path = '{}/saas-worker-stats'.format(self.Parameters.temp_node_path)
        saas_converter.Parameters.env = self.Parameters.env
        saas_converter.Parameters.secret_env = self.Parameters.secret_env

        saas_converter.save().enqueue()
        return saas_converter.id

    def get_standalone_indexer(self):
        logging.info('Starting get_standalone_indexer.')
        get_standalone_indexer_binary = SAAS_UTIL_GET_STANDALONE_INDEXER.find(attrs=dict(released='stable')).first()
        get_standalone_indexer_binary_path = str(sdk2.ResourceData(get_standalone_indexer_binary).path)
        get_standalone_indexer_cmd = [
            get_standalone_indexer_binary_path,
            '--service', 'ydo_statistics',
            '--ctype', self.Parameters.ctype,
        ]
        execute_cmd(
            cmd=get_standalone_indexer_cmd,
            log_name='ydo_get_standalone_indexer',
            error_message='Get standalone indexer failed'
        )

    def publish_stats_to_saas(self, env):
        logging.info('Starting standalone indexer.')
        standalone_indexer_cmd = [
            './standalone_indexer',
            'yt',
            'configs/rtyserver.conf-common',
            '--proxy', 'hahn',
            '--src', '{}/saas-worker-stats/{}'.format(self.Parameters.temp_node_path, self.get_input_date()),
            '--dst-dir', self.Parameters.saas_node_path,
            '--service', 'ydo_statistics',
            '--searchmap', 'configs/searchmap.json',
            '--verbose',
        ]
        if self.Parameters.publish:
            standalone_indexer_cmd.append('--publish')
            if self.Parameters.ctype == 'stable_kv':
                standalone_indexer_cmd.extend([
                    '--publish-path',
                    '{}/stable_kv/ydo_statistics'.format(self.Parameters.saas_node_path)
                ])
        execute_cmd(
            cmd=standalone_indexer_cmd,
            log_name='ydo_standalone_indexer',
            error_message='Standalone indexer failed',
            env=env
        )

    @staticmethod
    def check_task_on_success(task_id, task_name=''):
        child_status = sdk2.Task.find(id=task_id, children=True).first().status
        if child_status != ctt.Status.SUCCESS:
            raise Exception('Child task {} failed with status {}'.format(task_name, child_status))

    def accumulate_env_variables(self):
        env = os.environ.copy()
        for env_name, value in self.Parameters.env.iteritems():
            env[env_name] = str(value)
        for env_name, vault_name in self.Parameters.secret_env.iteritems():
            env[env_name] = sdk2.Vault.data(self.owner, vault_name)
        return env

    def on_execute(self):
        with self.memoize_stage.accumulate_worker_stats:
            self.Context.current_subtask_id = self.accumulate_worker_stats()
            raise sdk2.WaitTask(
                tasks=[self.Context.current_subtask_id],
                statuses=[ctt.Status.Group.FINISH, ctt.Status.Group.STOP, ctt.Status.Group.FAIL_ON_ANY_ERROR],
                wait_all=True
            )

        with self.memoize_stage.check_worker_stats_task:
            self.check_task_on_success(task_id=self.Context.current_subtask_id, task_name='accumulate_worker_stats')

        with self.memoize_stage.prepare_stats_for_saas:
            self.Context.current_subtask_id = self.prepare_stats_for_saas()
            raise sdk2.WaitTask(
                tasks=[self.Context.current_subtask_id],
                statuses=[ctt.Status.Group.FINISH, ctt.Status.Group.STOP, ctt.Status.Group.FAIL_ON_ANY_ERROR],
                wait_all=True
            )

        with self.memoize_stage.check_saas_preparator_task:
            self.check_task_on_success(task_id=self.Context.current_subtask_id, task_name='prepare_stats_for_saas')

        with self.memoize_stage.get_standalone_indexer:
            self.get_standalone_indexer()

        with self.memoize_stage.publish_stats_to_saas:
            env = self.accumulate_env_variables()
            self.publish_stats_to_saas(env)

        logging.info('Finished work.')

    def on_finish(self, prev_status, status):
        SolomonMixin.on_finish(self, prev_status, status, common_labels={'environment': self.Parameters.solomon_env})
        sdk2.Task.on_finish(self, prev_status, status)
