import os

import sandbox.common.types.task as ctt

from sandbox import sdk2

from sandbox.projects.geosearch.CleanupYtFolder import CleanupYtFolder

from sandbox.projects.ydo import get_now_utc, get_id_or_none, YdoDatabaseConfig, YdoKikimrExecutable
from sandbox.projects.ydo.backup.ToYt import YdoBackupToYt
from sandbox.projects.ydo.backup.MergeTables import YdoBackupMergeTables
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables


class YdoBackupConveyor(sdk2.Task):
    class Parameters(sdk2.Parameters):
        use_stable_resources = sdk2.parameters.Bool(
            "Use stable resources?",
            default=True,
        )
        with use_stable_resources.value[False]:
            kikimr_executable_resource = sdk2.parameters.Resource(
                "Kikimr",
                resource_type=YdoKikimrExecutable,
                required=True,
            )

            ydo_database_config = sdk2.parameters.Resource(
                "Database config",
                resource_type=YdoDatabaseConfig,
                required=True,
            )

        tables = sdk2.parameters.List(
            "Table to folder (space separated)",
            value_type=sdk2.parameters.String,
            required=True,
        )

        with sdk2.parameters.Group("YT parameters") as yt_block:
            yt_vault_token = sdk2.parameters.String("Your yt token name in vault", default="yt-token", required=True)
            with sdk2.parameters.RadioGroup("Host") as yt_host:
                yt_host.values["hahn"] = yt_host.Value(value="Hahn", default=True)
                yt_host.values["banach"] = yt_host.Value(value="Banach")
                yt_host.values["arnold"] = yt_host.Value(value="Arnold")

        history_size = sdk2.parameters.Integer('History size', default=5, required=True)
        merge_folder = sdk2.parameters.String('Folder for merged tables', required=True)
        links_folder = sdk2.parameters.String('Folder for links', required=True)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def init_context(self):
        self.Context.next_index = 0
        self.Context.tables = [(table_name, folder_path) for table_name, folder_path in map(lambda s: s.split(" "), self.Parameters.tables)]
        self.Context.result_tables = dict()
        self.Context.backup_tasks = list()

    def run_backup(self, table, folder_path):
        not_utc = str(get_now_utc())
        result_table = os.path.join(folder_path, not_utc)
        link_table = os.path.join(self.Parameters.links_folder, table)
        self.Context.result_tables[result_table] = link_table
        task = YdoBackupToYt(
            self,
            description="Backup {} for {}".format(table, self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            use_stable_resources=self.Parameters.use_stable_resources,
            kikimr_executable_resource=get_id_or_none(self.Parameters.kikimr_executable_resource),
            ydo_database_config=get_id_or_none(self.Parameters.ydo_database_config),
            table=table,
            yt_vault_token=self.Parameters.yt_vault_token,
            yt_host=self.Parameters.yt_host,
            yt_table=result_table,
        )

        self.Context.backup_tasks.append(task.id)

        task.enqueue()

        self.Context.next_index += 1

        raise sdk2.WaitTask(
            task.id,
            (ctt.Status.SUCCESS, ctt.Status.FAILURE, ctt.Status.TIMEOUT, ctt.Status.EXCEPTION, ctt.Status.NO_RES),
            wait_all=True
        )

    def links(self):
        for backup_task in self.Context.backup_tasks:
            if sdk2.Task[backup_task].status != ctt.Status.SUCCESS:
                return
        task = YdoBackupLinkTables(
            self,
            description='Link tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=self.Parameters.yt_host,
            yt_vault_token=self.Parameters.yt_vault_token,
            yt_tables=self.Context.result_tables
        )

        task.enqueue()

        raise sdk2.WaitTask([task.id], ctt.Status.Group.SUCCEED, wait_all=True)

    def merge_folders(self):
        tasks = [
            YdoBackupMergeTables(
                self,
                description='Merge old tables for task {}'.format(self.id),
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                yt_host=self.Parameters.yt_host,
                yt_vault_token=self.Parameters.yt_vault_token,
                path=folder_path,
                history_size=self.Parameters.history_size,
                merged_path=os.path.join(self.Parameters.merge_folder, table_name)
            )
            for table_name, folder_path in self.Context.tables
        ]
        for task in tasks:
            task.enqueue()

        raise sdk2.WaitTask([task.id for task in tasks], ctt.Status.Group.SUCCEED, wait_all=True)

    def clean_folders(self):
        tasks = [
            CleanupYtFolder(
                self,
                description='Remove old tables for task {}'.format(self.id),
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                yt_host=self.Parameters.yt_host,
                yt_vault_token=self.Parameters.yt_vault_token,
                path=folder_path,
                history_size=self.Parameters.history_size,
            )
            for _, folder_path in self.Context.tables
        ]
        for task in tasks:
            task.enqueue()

        raise sdk2.WaitTask([task.id for task in tasks], ctt.Status.Group.SUCCEED, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.init_context:
            self.init_context()

        if self.Context.next_index < len(self.Context.tables):
            self.run_backup(*self.Context.tables[self.Context.next_index])

        with self.memoize_stage.links:
            self.links()

        with self.memoize_stage.merge_folders:
            self.merge_folders()

        with self.memoize_stage.clean_folders:
            self.clean_folders()
