import datetime as dt
import json
import os

import sandbox.common.types.task as ctt

from sandbox import sdk2

from sandbox.projects.geosearch.CleanupYtFolder import CleanupYtFolder

from sandbox.projects.ydo import get_id_or_none, YdoYdbExecutable, YdoDbManagementExecutable
from sandbox.projects.ydo.backup.KikimrBackup import YdoBackupInKikimr
from sandbox.projects.ydo.backup.ToYt2 import YdoBackupToYt2
from sandbox.projects.ydo.backup.MergeTables import YdoBackupMergeTables
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables


class YdoBackupConveyor2(sdk2.Task):
    class Parameters(sdk2.Parameters):
        expires = dt.timedelta(minutes=30)

        use_stable_resources = sdk2.parameters.Bool(
            "Use stable resources?",
            default=True,
        )
        with use_stable_resources.value[False]:
            ydb_executable = sdk2.parameters.Resource(
                "ydb",
                resource_type=YdoYdbExecutable,
                required=True,
            )

            ydo_db_executable = sdk2.parameters.Resource(
                "ydo_db",
                resource_type=YdoDbManagementExecutable,
                required=True,
            )

        server = sdk2.parameters.String("YDB endpoint", required=True)

        is_multi_tenant = sdk2.parameters.Bool(
            "Multi Tenant?",
            default=False,
        )
        with is_multi_tenant.value[True]:
            database = sdk2.parameters.String("database", required=True)
            ydb_vault_token = sdk2.parameters.String("Your ydb token name in vault", default="yt-token", required=True)

        database_folder = sdk2.parameters.String("Folder need to backup", required=True)
        backups_folder = sdk2.parameters.String("Backups storage", required=True)

        with sdk2.parameters.Group("YT parameters") as yt_block:
            yt_vault_token = sdk2.parameters.String("Your yt token name in vault", default="yt-token", required=True)
            with sdk2.parameters.RadioGroup("Host") as yt_host:
                yt_host.values["hahn"] = yt_host.Value(value="Hahn", default=True)
                yt_host.values["banach"] = yt_host.Value(value="Banach")
                yt_host.values["arnold"] = yt_host.Value(value="Arnold")
            yt_folder = sdk2.parameters.String("Yt folder", required=True)

        history_size = sdk2.parameters.Integer('History size', default=5, required=True)
        merge_folder = sdk2.parameters.String('Folder for merged tables', required=True)
        links_folder = sdk2.parameters.String('Folder for links', required=True)

        backups_tasks_batch = sdk2.parameters.Integer('Concurrent backups count', default=1, required=True)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        backup_task_id = None
        links_tables = dict()
        backup_tasks = list()
        tables = list()
        yt_tasks = list()
        current_yt_task = None

    def run_backup(self):
        task = YdoBackupInKikimr(
            self,
            description="Backup for {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            use_stable_resources=self.Parameters.use_stable_resources,
            ydo_db_executable=get_id_or_none(self.Parameters.ydo_db_executable),
            server=self.Parameters.server,
            is_multi_tenant=self.Parameters.is_multi_tenant,
            database=self.Parameters.database,
            ydb_vault_token=self.Parameters.ydb_vault_token,
            database_folder=self.Parameters.database_folder,
            backups_folder=self.Parameters.backups_folder,
        )

        self.Context.backup_task_id = task.id

        task.enqueue()

        raise sdk2.WaitTask(
            task.id,
            (ctt.Status.SUCCESS, ctt.Status.FAILURE, ctt.Status.TIMEOUT, ctt.Status.EXCEPTION, ctt.Status.NO_RES),
            wait_all=True
        )

    def store_to_yt_one_table(self, timestamp, table):
        basename = os.path.basename(table)
        yt_table = os.path.join(self.Parameters.yt_folder, basename, timestamp)
        task = YdoBackupToYt2(
            self,
            description="Backup to yt {} for {}".format(table, self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            use_stable_resources=self.Parameters.use_stable_resources,
            ydb_executable=get_id_or_none(self.Parameters.ydb_executable),
            ydo_db_executable=get_id_or_none(self.Parameters.ydo_db_executable),
            host=self.Parameters.server,
            is_multi_tenant=self.Parameters.is_multi_tenant,
            database=self.Parameters.database,
            ydb_vault_token=self.Parameters.ydb_vault_token,
            table=table,
            yt_vault_token=self.Parameters.yt_vault_token,
            yt_host=self.Parameters.yt_host,
            yt_table=yt_table
        )

        self.Context.links_tables[yt_table] = os.path.join(self.Parameters.links_folder, basename)
        self.Context.backup_tasks.append(task.id)
        self.Context.tables.append((basename, os.path.dirname(yt_table)))

        return task

    def generate_store_to_yt(self):
        backuped_json = json.loads(sdk2.Task[self.Context.backup_task_id].Parameters.json_result)
        timestamp = os.path.basename(backuped_json["dir_name"])

        tasks = list()

        for table in backuped_json["table_names"]:
            tasks.append(self.store_to_yt_one_table(timestamp, os.path.join(backuped_json["dir_name"], table)))

        for task in tasks:
            self.Context.yt_tasks.append(task.id)

        self.Context.current_yt_task = 0

    def run_store_to_yt(self):
        task_ids = list()
        try:
            for i in range(int(self.Parameters.backups_tasks_batch)):
                task_ids.append(self.Context.yt_tasks[self.Context.current_yt_task + i])
        except IndexError:
            if not task_ids:
                return

        tasks = [sdk2.Task[task_id] for task_id in task_ids]
        for task in tasks:
            task.enqueue()

        self.Context.current_yt_task += len(task_ids)
        raise sdk2.WaitTask(
            task_ids,
            (ctt.Status.SUCCESS, ctt.Status.FAILURE, ctt.Status.TIMEOUT, ctt.Status.EXCEPTION, ctt.Status.NO_RES),
            wait_all=True
        )

    def links(self):
        for backup_task in self.Context.backup_tasks:
            if sdk2.Task[backup_task].status != ctt.Status.SUCCESS:
                return
        task = YdoBackupLinkTables(
            self,
            description='Link tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=self.Parameters.yt_host,
            yt_vault_token=self.Parameters.yt_vault_token,
            yt_tables=self.Context.links_tables
        )

        task.enqueue()

        raise sdk2.WaitTask([task.id], ctt.Status.Group.SUCCEED, wait_all=True)

    def merge_folders(self):
        tasks = [
            YdoBackupMergeTables(
                self,
                description='Merge old tables for task {}'.format(self.id),
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                yt_host=self.Parameters.yt_host,
                yt_vault_token=self.Parameters.yt_vault_token,
                path=folder_path,
                history_size=self.Parameters.history_size,
                merged_path=os.path.join(self.Parameters.merge_folder, table_name)
            )
            for table_name, folder_path in self.Context.tables
        ]
        for task in tasks:
            task.enqueue()

        raise sdk2.WaitTask([task.id for task in tasks], ctt.Status.Group.SUCCEED, wait_all=True)

    def clean_folders(self):
        tasks = [
            CleanupYtFolder(
                self,
                description='Remove old tables for task {}'.format(self.id),
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                yt_host=self.Parameters.yt_host,
                yt_vault_token=self.Parameters.yt_vault_token,
                path=folder_path,
                history_size=self.Parameters.history_size,
            )
            for _, folder_path in self.Context.tables
        ]
        for task in tasks:
            task.enqueue()

        raise sdk2.WaitTask([task.id for task in tasks], ctt.Status.Group.SUCCEED, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.run_backup:
            self.run_backup()

        with self.memoize_stage.generate_store_to_yt:
            self.generate_store_to_yt()

        self.run_store_to_yt()

        with self.memoize_stage.links:
            self.links()

        with self.memoize_stage.merge_folders:
            self.merge_folders()

        with self.memoize_stage.clean_folders:
            self.clean_folders()
