import os
import json
import logging
import tempfile

import sandbox.common.types.task as ctt

from sandbox import sdk2

from sandbox.projects.common import file_utils as fu

from sandbox.projects.yql.RunYQL2 import RunYQL2

from sandbox.projects.ydo import get_last_released_resource, YdoDatabaseConfig, YdoKikimrExecutable, execute_cmd
from sandbox.projects.ydo.config_operations import (
    get_table_path,
    get_database_host,
    get_database_host_with_port,
    get_table_types,
    get_primary_columns,
)

create_table_message = """
ModifyScheme {{
    WorkingDir : '{kikimr_home}'
    OperationType : ESchemeOpCreateTable
    CreateTable : {{
        Name : '{table_name}'
        {columns}
        {key_columns}
    }}
}}
"""

column_message = """Columns : {{
    Name : '{name}'
    Type : '{type}'
}}"""


def format_column_message(column):
    return column_message.format(name=column["name"], type=column["type"].capitalize())


key_column_message = "KeyColumnNames : '{name}'"

host_to_endpoints = {
    "ydb-services.yandex.net": [
        "ydb-services-vla-000.search.yandex.net",
        "ydb-services-vla-001.search.yandex.net",
        "ydb-services-vla-002.search.yandex.net",
        "ydb-services-myt-000.search.yandex.net",
        "ydb-services-myt-001.search.yandex.net",
        "ydb-services-myt-002.search.yandex.net",
        "ydb-services-sas-000.search.yandex.net",
        "ydb-services-sas-001.search.yandex.net",
        "ydb-services-sas-002.search.yandex.net",
    ]
}


def format_key_column_message(column_name):
    return key_column_message.format(name=column_name)


class YdoBackupToKikimr(sdk2.Task):
    class Parameters(sdk2.Parameters):
        use_stable_resources = sdk2.parameters.Bool(
            "Use stable resources?",
            default=True,
        )
        with use_stable_resources.value[False]:
            kikimr_executable_resource = sdk2.parameters.Resource(
                "Kikimr",
                resource_type=YdoKikimrExecutable,
                required=True,
            )

            ydo_database_config = sdk2.parameters.Resource(
                "Database config",
                resource_type=YdoDatabaseConfig,
                required=True,
            )

        table = sdk2.parameters.String(
            "Table name",
            required=True,
        )

        with sdk2.parameters.Group("YT parameters") as yt_block:
            yql_vault_token = sdk2.parameters.String("Your yql token name in vault", default="YQL_TOKEN", required=True)
            with sdk2.parameters.RadioGroup("Host") as yt_host:
                yt_host.values["hahn"] = yt_host.Value(value="Hahn", default=True)
                yt_host.values["banach"] = yt_host.Value(value="Banach")
            yt_table = sdk2.parameters.String("Result table", required=True)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def init_context(self):
        config_data = get_last_released_resource(YdoDatabaseConfig, self.Parameters.use_stable_resources, self.Parameters.ydo_database_config)
        with open(str(config_data.path)) as i:
            self.Context.config = json.load(i)

    def create_kikimr_table(self):
        kikimr_path = str(get_last_released_resource(
            YdoKikimrExecutable,
            condition=self.Parameters.use_stable_resources,
            default=self.Parameters.kikimr_executable_resource,
            error_msg="YdoKikimrExecutable not founded",
        ).path)

        columns = "\n".join(map(format_column_message, get_table_types(self.Context.config, self.Parameters.table)))
        key_columns = "\n".join(map(format_key_column_message, get_primary_columns(self.Context.config, self.Parameters.table)))
        kikimr_home, table_name = os.path.split(get_table_path(self.Context.config, self.Parameters.table))

        message_file = tempfile.NamedTemporaryFile().name
        message = create_table_message.format(
            kikimr_home=kikimr_home,
            table_name=table_name,
            columns=columns,
            key_columns=key_columns
        )

        fu.write_file(message_file, message)
        logging.info("Create table message:" + message)

        execute_cmd(
            [
                kikimr_path,
                "-s", get_database_host_with_port(self.Context.config),
                "db", "schema", "execute",
                message_file
            ],
            "kikimr db schema execute create_table",
            "Create table failed",
        )

    def recover_table(self):
        with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), "recover_table.sql")) as query_file:
            query = query_file.read()

        placeholders = {
            "%CLUSTER%": self.Parameters.yt_host,
            "%KIKIMR_HOSTS%": "\"{}\"".format("\", \"".join(host_to_endpoints[get_database_host(self.Context.config)])),
            "%KIKIMR_TABLE%": get_table_path(self.Context.config, self.Parameters.table),
            "%BACKUP_TABLE%": self.Parameters.yt_table,
            "%SCHEMA%": ", ".join(map(lambda column: column["name"], get_table_types(self.Context.config, self.Parameters.table)))
        }

        task = RunYQL2(
            self,
            description="Push backup for task {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            query=query,
            custom_placeholders=placeholders,
            trace_query=True,
            yql_token_vault_name=self.Parameters.yql_vault_token
        )
        task.enqueue()

        raise sdk2.WaitTask([task.id], ctt.Status.Group.SUCCEED, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.init_context:
            self.init_context()

        with self.memoize_stage.create_kikimr_table:
            self.create_kikimr_table()

        with self.memoize_stage.recover_table:
            self.recover_table()
