import os
import json
import logging
import tempfile

from sandbox import sdk2

from sandbox.projects.common import file_utils as fu

from sandbox.projects.ydo import get_last_released_resource, YdoDatabaseConfig, YdoKikimrExecutable, get_now_utc, execute_cmd
from sandbox.projects.ydo.config_operations import get_table_path, get_database_host_with_port, get_table_types, get_backup_folder

from sandbox.projects.ydo.backup import YdoBackupBaseTask

copy_deduper = """
ModifyScheme {{
    WorkingDir : '{backup_folder}'
    OperationType : ESchemeOpCreateTable
    CreateTable : {{
        Name : '{backup_table_name}'
        CopyFromTable: '{table}'
    }}
}}
"""

backup_deduper = """
ModifyScheme {{
    WorkingDir: '{backup_folder}'
    OperationType: ESchemeOpBackup
    Backup: {{
        TableName: '{backup_table_name}'
        BackupToYTSettings: {{
            BrokerHost: '{yt_cluster}.yt.yandex.net'
            BrokerPort: 80
            DestinationTablePattern: '<append=true>{yt_table}'
            Token: '{yt_token}'
        }}
    }}
}}
"""


class YdoBackupToYt(YdoBackupBaseTask):
    class Parameters(sdk2.Parameters):
        use_stable_resources = sdk2.parameters.Bool(
            "Use stable resources?",
            default=True,
        )
        with use_stable_resources.value[False]:
            kikimr_executable_resource = sdk2.parameters.Resource(
                "Kikimr",
                resource_type=YdoKikimrExecutable,
                required=True,
            )

            ydo_database_config = sdk2.parameters.Resource(
                "Database config",
                resource_type=YdoDatabaseConfig,
                required=True,
            )

        table = sdk2.parameters.String(
            "Table name",
            required=True,
        )

        with sdk2.parameters.Group("YT parameters") as yt_block:
            yt_vault_token = sdk2.parameters.String("Your yt token name in vault", default="yt-token", required=True)
            with sdk2.parameters.RadioGroup("Host") as yt_host:
                yt_host.values["hahn"] = yt_host.Value(value="Hahn", default=True)
                yt_host.values["banach"] = yt_host.Value(value="Banach")
                yt_host.values["arnold"] = yt_host.Value(value="Arnold")
            yt_table = sdk2.parameters.String("Result table", required=True)

    def init_context(self):
        config_data = get_last_released_resource(YdoDatabaseConfig, self.Parameters.use_stable_resources, self.Parameters.ydo_database_config)
        with open(str(config_data.path)) as i:
            self.Context.config = json.load(i)

        self.Context.kikimr_path = str(get_last_released_resource(
            YdoKikimrExecutable,
            condition=self.Parameters.use_stable_resources,
            default=self.Parameters.kikimr_executable_resource,
            error_msg="YdoKikimrExecutable not founded",
        ).path)

        self.Context.kikimr_table_name = "{}_{}_from_sandbox_{}".format(self.Parameters.table, get_now_utc(), self.id)

    def copy_table(self):
        message_file = tempfile.NamedTemporaryFile().name
        message = copy_deduper.format(
            backup_folder=get_backup_folder(self.Context.config),
            table=get_table_path(self.Context.config, self.Parameters.table),
            backup_table_name=self.Context.kikimr_table_name
        )

        fu.write_file(message_file, message)
        logging.info("Copy message:" + message)

        execute_cmd(
            [
                self.Context.kikimr_path,
                "-s", get_database_host_with_port(self.Context.config),
                "db", "schema", "execute",
                message_file
            ],
            "kikimr db schema execute copy",
            "Copy failed",
        )

    def backup_table(self):
        yt_token, yt_host = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token), self.Parameters.yt_host

        # create table
        import yt.wrapper as yt
        yt.config["token"], yt.config["proxy"]["url"] = yt_token, yt_host

        assert not yt.exists(self.Parameters.yt_table)
        yt.create(
            "table", self.Parameters.yt_table,
            attributes=dict(schema=get_table_types(self.Context.config, self.Parameters.table), optimize_for="scan")
        )
        assert yt.exists(self.Parameters.yt_table)

        # make backup request
        message_parameters = dict(
            backup_folder=get_backup_folder(self.Context.config),
            backup_table_name=self.Context.kikimr_table_name,
            yt_cluster=yt_host,
            yt_token=yt_token,
            yt_table=self.Parameters.yt_table,
        )

        message_file = tempfile.NamedTemporaryFile().name
        message = backup_deduper.format(**message_parameters)
        message_parameters["yt_token"] = "SECRET"
        fake_message = backup_deduper.format(**message_parameters)

        fu.write_file(message_file, message)
        logging.info("Backup message:" + fake_message)

        execute_cmd(
            [
                self.Context.kikimr_path,
                "-s", get_database_host_with_port(self.Context.config),
                "db", "schema", "execute",
                message_file
            ],
            "kikimr db schema execute backup",
            "Backup failed",
        )

        execute_cmd(
            [
                self.Context.kikimr_path,
                "-s", get_database_host_with_port(self.Context.config),
                "db", "schema", "drop",
                os.path.join(get_backup_folder(self.Context.config), self.Context.kikimr_table_name)
            ],
            "kikimr db schema drop backup_table",
            "Backup failed",
        )

    def on_execute(self):
        self.init_context()

        self.copy_table()

        self.backup_table()
