import os

from sandbox import sdk2

from sandbox.projects.ydo import get_last_released_resource, execute_cmd, YdoDbManagementExecutable, YdoYdbExecutable


class YdoBackupToYt2(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        use_stable_resources = sdk2.parameters.Bool(
            "Use stable resources?",
            default=True,
        )
        with use_stable_resources.value[False]:
            ydb_executable = sdk2.parameters.Resource(
                "ydb",
                resource_type=YdoYdbExecutable,
                required=True,
            )

            ydo_db_executable = sdk2.parameters.Resource(
                "ydo_db",
                resource_type=YdoDbManagementExecutable,
                required=True,
            )

        host = sdk2.parameters.String(
            "Kikimr host",
            required=True,
        )

        is_multi_tenant = sdk2.parameters.Bool(
            "Multi Tenant?",
            default=False,
        )
        with is_multi_tenant.value[True]:
            database = sdk2.parameters.String("database", required=True)
            ydb_vault_token = sdk2.parameters.String("Your ydb token name in vault", default="yt-token", required=True)

        table = sdk2.parameters.String(
            "Table path",
            required=True,
        )

        with sdk2.parameters.Group("YT parameters") as yt_block:
            yt_vault_token = sdk2.parameters.String("Your yt token name in vault", default="yt-token", required=True)
            with sdk2.parameters.RadioGroup("Host") as yt_host:
                yt_host.values["hahn"] = yt_host.Value(value="Hahn", default=True)
                yt_host.values["arnold"] = yt_host.Value(value="Arnold")
            yt_table = sdk2.parameters.String("Result table", required=True)

    def on_execute(self):
        ydb = str(
            get_last_released_resource(
                YdoYdbExecutable,
                condition=self.Parameters.use_stable_resources,
                default=self.Parameters.ydb_executable,
                error_msg="YdoYdbExecutable not founded",
            ).path
        )
        ydo_db = str(
            get_last_released_resource(
                YdoDbManagementExecutable,
                condition=self.Parameters.use_stable_resources,
                default=self.Parameters.ydo_db_executable,
                error_msg="YdoDbManagementExecutable not founded",
            ).path
        )
        env = os.environ.copy()
        env["YT_TOKEN"] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
        cmd = [
            ydo_db,
            "--server", self.Parameters.host,
            "--debug"
        ]

        if self.Parameters.is_multi_tenant:
            cmd += ["--database", self.Parameters.database]
            env["YDB_TOKEN"] = sdk2.Vault.data(self.owner, self.Parameters.ydb_vault_token)

        cmd += [
            "backup_to_yt",
            "--ydb-cli", ydb,
            "--kikimr-table", self.Parameters.table,
            "--yt-cluster", self.Parameters.yt_host,
            "--yt-table", self.Parameters.yt_table,
        ]

        execute_cmd(
            cmd,
            "backup_to_yt",
            "Failed",
            fail_on_error=False,
            env=env,
        )
