import json

from sandbox import sdk2

from sandbox.projects.ydo.resource_types import YdoArbitragePenaltyConfig
from sandbox.projects.ydo import ydo_releasers
from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoBanBadExternalWorkersExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoBanBadExternalWorkers(sdk2.Task, SolomonMixinV2):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(
        generate_view(
            ydo_ban_bad_external_workers=YdoBanBadExternalWorkersExecutable, check_owner=True, with_environ=True
        )
    ):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)
        with sdk2.parameters.Group('YT_TABLES'):
            yt_external_workers = sdk2.parameters.String('YT external_workers path', required=True)
            yt_orders = sdk2.parameters.String('YT orders path', required=True)
            yt_orders_aux = sdk2.parameters.String('YT orders_aux path', required=True)
            yt_ratings = sdk2.parameters.String('YT ratings path', required=True)
            yt_service_complaints = sdk2.parameters.String('YT service_complaints path', required=True)
            yt_service_incidents = sdk2.parameters.String('YT service_incidents path', required=True)
            yt_order_partners = sdk2.parameters.String('YT order_partners path', required=False)
            yt_external_worker_attributes = sdk2.parameters.String('YT external_worker_attributes path', required=True)
            yt_order_events_history = sdk2.parameters.String('YT order_events_history path', required=False)
            yt_erp_employees = sdk2.parameters.String('YT erp_employees path', required=False)
            yt_erp_organizations = sdk2.parameters.String('YT erp_organizations path', required=False)
            yt_erp_calendars_dir = sdk2.parameters.String('YT erp_calendars directory path', required=False)

        rating_threshold = sdk2.parameters.Float('rating threshold', default=4.5)
        skip_reviews_older_than_ts = sdk2.parameters.Integer('Min rating timestamp for trial period', required=True)
        max_count_incidents = sdk2.parameters.Dict('Max count incidents for worker to be blocked', default={})
        schedule_refusals_settings = sdk2.parameters.JSON(
            'Ban settings for schedule refusals ban reason', required=False
        )
        utilization_settings = sdk2.parameters.JSON(
            'Utilization settings for low utilization ban reason', required=False
        )
        work_time_settings = sdk2.parameters.JSON('Work time settings for short work time ban reason', required=False)
        direct_work_arb_enable_settings = sdk2.parameters.JSON(
            'Enable settings for direct work with workers arbitrage', required=False
        )

        robot_login = sdk2.parameters.String('Robot login', default='robot-services')
        tvm_id = sdk2.parameters.Integer('TVM id', required=True)
        ydo_intapi_host = sdk2.parameters.String('ydo intapi host', required=True)
        samsara_api_host = sdk2.parameters.String('Samsara API host', required=True)
        adminka_home_url = sdk2.parameters.String('Adminka home url', required=True)
        backend_home_url = sdk2.parameters.String('Backend home url', required=False)
        samsara_queue_ids = sdk2.parameters.Dict('Samsara queue IDs', required=False)
        oauth_token_name = sdk2.parameters.String('oauth token name', required=True)
        yang_pool_ids = sdk2.parameters.Dict('Yang pool IDs', required=False)
        yang_api_host = sdk2.parameters.String('Yang API host', required=False)

        logging_config = sdk2.parameters.String('LoggingConfig')

    def add_param_to_cmd(self, cmd, name, parameter, exclude=None):
        if parameter and parameter not in (exclude or []):
            cmd += [name, parameter]

    def on_execute(self):
        get_penalty_config = YdoArbitragePenaltyConfig.find(attrs=dict(released='stable')).first()
        get_penalty_config_path = str(sdk2.ResourceData(get_penalty_config).path)

        cmd = [
            self.Parameters.ydo_ban_bad_external_workers_path,
            '--kikimr-host',
            self.Parameters.ydb_host,
            '--kikimr-home',
            self.Parameters.ydb_home,
            '--kikimr-database',
            self.Parameters.ydb_database,
            '--kikimr-port',
            str(self.Parameters.ydb_port),
            '--external_workers',
            self.Parameters.yt_external_workers,
            '--orders',
            self.Parameters.yt_orders,
            '--orders_aux',
            self.Parameters.yt_orders_aux,
            '--ratings',
            self.Parameters.yt_ratings,
            '--service_complaints',
            self.Parameters.yt_service_complaints,
            '--service_incidents',
            self.Parameters.yt_service_incidents,
            '--order_partners',
            self.Parameters.yt_order_partners,
            '--external_worker_attributes',
            self.Parameters.yt_external_worker_attributes,
            '--order_events_history',
            self.Parameters.yt_order_events_history,
            '--erp_employees',
            self.Parameters.yt_erp_employees,
            '--erp_organizations',
            self.Parameters.yt_erp_organizations,
            '--penalty_config',
            get_penalty_config_path,
            '--erp_calendars_dir',
            self.Parameters.yt_erp_calendars_dir,
        ]

        self.add_param_to_cmd(cmd, '--rating_threshold', str(self.Parameters.rating_threshold))
        self.add_param_to_cmd(cmd, '--skip_reviews_older_than_ts', str(self.Parameters.skip_reviews_older_than_ts))
        self.add_param_to_cmd(cmd, '--max_count_incidents', json.dumps(self.Parameters.max_count_incidents))
        self.add_param_to_cmd(
            cmd,
            '--schedule_refusals_settings',
            json.dumps(self.Parameters.schedule_refusals_settings),
            exclude=['null'],
        )
        self.add_param_to_cmd(
            cmd,
            '--utilization_settings',
            json.dumps(self.Parameters.utilization_settings),
            exclude=['null'],
        )
        self.add_param_to_cmd(
            cmd,
            '--work_time_settings',
            json.dumps(self.Parameters.work_time_settings),
            exclude=['null'],
        )
        self.add_param_to_cmd(
            cmd,
            '--direct_work_arb_enable_settings',
            json.dumps(self.Parameters.direct_work_arb_enable_settings),
            exclude=['null'],
        )

        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)
        self.add_param_to_cmd(cmd, '--kikimr-retry-count', str(self.Parameters.ydb_retry_count))
        self.add_param_to_cmd(cmd, '--kikimr-delay', str(self.Parameters.ydb_delay))
        self.add_param_to_cmd(cmd, '--kikimr-backoff', str(self.Parameters.ydb_backoff))
        self.add_param_to_cmd(cmd, '--kikimr-limit', str(self.Parameters.ydb_limit))

        self.add_param_to_cmd(cmd, '--robot_login', self.Parameters.robot_login)
        self.add_param_to_cmd(cmd, '--tvm-id', str(self.Parameters.tvm_id))
        self.add_param_to_cmd(cmd, '--ydo-intapi-host', self.Parameters.ydo_intapi_host)
        self.add_param_to_cmd(cmd, '--samsara_api_host', self.Parameters.samsara_api_host)
        self.add_param_to_cmd(cmd, '--adminka_home_url', self.Parameters.adminka_home_url)
        self.add_param_to_cmd(cmd, '--backend_home_url', self.Parameters.backend_home_url)
        self.add_param_to_cmd(cmd, '--oauth_token_name', self.Parameters.oauth_token_name)
        self.add_param_to_cmd(cmd, '--samsara_queue_ids', json.dumps(self.Parameters.samsara_queue_ids), exclude=['{}'])
        self.add_param_to_cmd(cmd, '--yang_pool_ids', json.dumps(self.Parameters.yang_pool_ids), exclude=['{}'])
        self.add_param_to_cmd(cmd, '--yang_api_host', self.Parameters.yang_api_host)

        execute_cmd(
            cmd,
            'ydo_ban_bad_external_workers',
            'ydo_ban_bad_external_workers failed',
            env=self.Parameters.get_environ(),
        )

    def on_break(self, *args, **kwargs):
        SolomonMixinV2.on_break(self, *args, **kwargs)
        super(YdoBanBadExternalWorkers, self).on_break(*args, **kwargs)

    def on_finish(self, *args, **kwargs):
        SolomonMixinV2.on_finish(self, *args, **kwargs)
        super(YdoBanBadExternalWorkers, self).on_finish(*args, **kwargs)
