import logging
import time
import requests
from requests.auth import AuthBase
import yaml

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2
from sandbox.projects.ydo.default_multislot_task import YdoMutlihostTask

logger = logging.getLogger(__name__)


class OAuth(AuthBase):
    def __init__(self, token):
        self._header = 'OAuth ' + token

    def __call__(self, r):
        r.headers['Authorization'] = self._header
        return r


class RpsLimiterClient(object):
    base_url = 'https://rpslimiter.z.yandex-team.ru/api/rpslimiter.Model/'

    def __init__(self, installation, section, token):
        self.installation = installation
        self.section = section
        self.auth = OAuth(token)

    def _request(self, http_method, method, **kwargs):
        kwargs.setdefault('timeout', 60)
        kwargs['auth'] = self.auth
        url = self.base_url + method

        logger.info('%s %s', http_method.upper(), url)
        if 'params' in kwargs:
            logger.info('PARAMS: %s', kwargs['params'])
        if 'json' in kwargs:
            logger.info('JSON: %s', kwargs['json'])
        N = 5
        for i in range(N):
            try:
                response = requests.request(http_method, url, **kwargs)
                logger.info('http_code = %s, content length = %s', response.status_code, len(response.content))
                if 'x-req-id' in response.headers:
                    logger.info('Request id: "%s"', response.headers['x-req-id'])
                if response.headers.get('grpc-status', -1) != 0:
                    logger.info('Grpc status: %s', response.headers.get('grpc-status', -1))
                    logger.info('Message: %s', response.headers.get('grpc-message', ''))

                response.raise_for_status()
                result = response.json()
                if result.get('error'):
                    raise RuntimeError(result['error'])
                return result
            except:
                logger.exception('Request failed')
                if i + 1 == N:
                    raise
                logger.exception('Wait before next request')
                time.sleep(15)

    def get_quota(self, quota_name):
        result = self._request('POST', 'fetchConfig', json={'parent': self.installation, 'id': self.section})
        quota_configs = yaml.safe_load(result['content']['quotas'])
        return quota_configs[quota_name]['quota']

    def update_quota(self, quota_name, new_value, comment=None):
        result = self._request('POST', 'fetchConfig', json={'parent': self.installation, 'id': self.section})
        result.pop('author', None)
        result.pop('unixtimeMcs', None)
        quota_configs = yaml.safe_load(result['content']['quotas'])
        quota_configs[quota_name]['quota'] = new_value
        result['content']['quotas'] = yaml.dump(quota_configs)
        if comment is not None:
            result['content']['quotas'] = '# {}\n{}'.format(comment.strip(), result['content']['quotas'])

        self._request('POST', 'updateConfig', json=result)


class YdoChangeRpsLimiterQuota(YdoMutlihostTask, SolomonMixinV2):
    class Requirements(YdoMutlihostTask.Requirements):
        environments = (environments.PipEnvironment('yasmapi', use_wheel=True),)

    class Parameters(sdk2.Parameters):

        rps_limiter_installation = sdk2.parameters.String('Rps limiter installation', default='main', required=True)
        rps_limiter_section = sdk2.parameters.String('Rps limiter section', required=True)

        rps_limiter_quota = sdk2.parameters.String('Rps limiter quota', required=True)
        rps_limiter_quota_limit = sdk2.parameters.Integer('Max rps limiter quota limit', required=True)

        golovan_rps_signal = sdk2.parameters.String('Golovan signal with rps', required=True)
        window_size_m = sdk2.parameters.Integer('Window size in minutes', default=10, required=True)

        golovan_rps_signal_by_geo_template = sdk2.parameters.String('Golovan signal with rps by geo')
        geos = sdk2.parameters.List('Geos', value_type=sdk2.parameters.String)

        oauth_token_secret = sdk2.parameters.YavSecret('Yav secret for oauth token', required=True)
        oauth_token_key = sdk2.parameters.String('Name of record in secret', default='INFRA_AUTH_TOKEN', required=True)

    def get_rps_percentile(self, signal, window_size, percentile=0.95):
        from yasmapi import GolovanRequest

        host = 'ASEARCH'
        to_time = time.time() - 60
        from_time = to_time - window_size * 60
        period = 5
        rps = []
        for _, values in GolovanRequest(host, period, from_time, to_time, [signal]):
            rps.append((values[signal] + period - 1) / period)
        rps.sort()
        return rps[max(0, int(len(rps) * percentile) - 1)]

    def get_current_rps(self):
        return self.get_rps_percentile(
            self.Parameters.golovan_rps_signal,
            self.Parameters.window_size_m,
            percentile=0.95,
        )

    def get_multiplier_by_closed_geo(self):
        if not self.Parameters.golovan_rps_signal_by_geo_template:
            return 1
        geos_rps = {
            geo: self.get_rps_percentile(
                self.Parameters.golovan_rps_signal_by_geo_template.format(geo=geo),
                window_size=2,
                percentile=0.5,
            )
            for geo in self.Parameters.geos
        }
        logger.info('Rps by geo: %s', geos_rps)
        geos_rps = list(geos_rps.values())
        geos_rps.sort()
        disabled_location = 0
        for geo_rps in geos_rps:
            if 1.0 * geo_rps / (geos_rps[-1] + 0.01) < 0.9:
                disabled_location += 1
        return 1 - 1.0 * disabled_location / len(geos_rps)

    def calculate_next_rps_limit(self, current_rps_limit, current_rps):
        min_limit = 10
        minimal_step = 5
        multiplier_by_closed_geo = self.get_multiplier_by_closed_geo()
        max_limit = int(self.Parameters.rps_limiter_quota_limit * multiplier_by_closed_geo)
        logger.info('Max limit multiplier %0.2f. Max rps limit %s', multiplier_by_closed_geo, max_limit)
        logger.info('Current rps %s, current rps limit %s', current_rps, current_rps_limit)
        target_limit = min(max_limit, max(min_limit, current_rps))
        logger.info('Target limit %s', target_limit)
        if current_rps_limit >= target_limit:
            diff = current_rps_limit - target_limit
            step = diff / 2
            step = max(step, minimal_step)
            next_rps_limit = max(target_limit, current_rps_limit - step)
        else:
            diff = target_limit - current_rps_limit
            step = diff / 3
            step = max(step, minimal_step)
            next_rps_limit = min(target_limit, current_rps_limit + step)
        next_rps_limit = min(max_limit, next_rps_limit)
        logger.info('Next rps limit %s', next_rps_limit)
        return next_rps_limit

    def on_execute(self):
        token = self.Parameters.oauth_token_secret.data()[self.Parameters.oauth_token_key]
        client = RpsLimiterClient(self.Parameters.rps_limiter_installation, self.Parameters.rps_limiter_section, token)
        current_rps_limit = client.get_quota(self.Parameters.rps_limiter_quota)
        current_rps = self.get_current_rps()
        next_rps_limit = self.calculate_next_rps_limit(current_rps_limit, current_rps)
        if self.scheduler and self.scheduler >= 0:  # < 0 manual runs
            comment = 'Modified automatically by sandbox scheduler {}'.format(str(self.scheduler))
        else:
            comment = 'Modified by sandbox task {}'.format(str(self.id))
        client.update_quota(self.Parameters.rps_limiter_quota, next_rps_limit, comment=comment)

    def on_break(self, *args, **kwargs):
        SolomonMixinV2.on_break(self, *args, **kwargs)
        super(YdoChangeRpsLimiterQuota, self).on_break(*args, **kwargs)

    def on_finish(self, *args, **kwargs):
        SolomonMixinV2.on_finish(self, *args, **kwargs)
        super(YdoChangeRpsLimiterQuota, self).on_finish(*args, **kwargs)
