# coding: utf-8
import datetime
import logging
import os
from collections import defaultdict

import sandbox.sdk2 as sdk2
from sandbox.sandboxsdk import environments
from dateutil import parser as time_parser
from sandbox.projects.ydo.parameters_view import generate_view_for_yt
from sandbox.projects.common import task_env


logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


def check_and_gen_dates(from_date, to_date):
    current_date = time_parser.parse(from_date).date()
    last_date = time_parser.parse(to_date).date()
    diff = last_date - current_date

    if diff.days < 0 or diff.days > 365:
        raise Exception("Incorrect days diff: {}".format(diff))

    result = []
    while current_date <= last_date:
        result.append(str(current_date))
        current_date = (current_date + datetime.timedelta(days=1))
    return result


def get_connections_from_one_table(yt, table):
    for connection_row in yt.read_table(table):
        worker_id = connection_row['worker_id']
        puid = connection_row['puid']
        if worker_id and puid and puid != 'None':
            yield worker_id, puid


def get_all_connections(yt, connections_directory):
    worker_id2puids = defaultdict(lambda: defaultdict(int))
    for table_name in yt.list(connections_directory):
        logger.info('Reading connections for table {}'.format(table_name))
        for worker_id, puid in get_connections_from_one_table(yt, os.path.join(connections_directory, table_name)):
            worker_id2puids[worker_id][puid] += 1

    return worker_id2puids


def update_connections(yt, connections_directory, connection_map_table):
    logger.info('Reading all connections')
    new_worker_id2puids = get_all_connections(yt, connections_directory)

    backup_connections_table = connection_map_table + '_backup'
    logger.info('Renaming connections table: {} -> {}'.format(connection_map_table, backup_connections_table))

    if not yt.exists(connection_map_table):
        logger.warning('Table {} does not exist. Creating empty table'.format(connection_map_table))
        yt.write_table(connection_map_table, [])

    yt.move(connection_map_table, backup_connections_table)

    try:
        logger.info('Writing updated connections')
        result_rows = [{'worker_id': worker_id, 'puids': dict(puid2count)} for worker_id, puid2count in new_worker_id2puids.items()]
        yt.write_table(connection_map_table, result_rows)

        logger.info('Removing backup connections table {}'.format(backup_connections_table))
        yt.remove(backup_connections_table)

    except Exception as e:
        logger.info('Error: {}. Rename table back: {} -> {}'.format(e, backup_connections_table, connection_map_table))
        if yt.exists(connection_map_table):
            yt.remove(connection_map_table)
        yt.move(backup_connections_table, connection_map_table)


class YdoCollectConnections(sdk2.Task):
    """
        Сбор коннектов из выбранного интервала дат в одну таблицу
    """

    class Requirements(task_env.TinyRequirements):
        ram = 1024
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    class Parameters(generate_view_for_yt(
        check_owner=True,
        with_environ=True,
    )):
        source_connections_directory = sdk2.parameters.String(
            'Source connections directory path',
            required=True
        )

        destination_connections = sdk2.parameters.String(
            'Destination connections table path',
            required=True
        )

        yt_cluster = sdk2.parameters.String(
            'YT cluster',
            default_value='hahn'
        )

    def on_execute(self):
        import yt.wrapper as yt

        yt.config['token'] = sdk2.Vault.data(self.owner, 'yt-token')
        yt.config['proxy']['url'] = '{}.yt.yandex.net'.format(self.Parameters.yt_cluster)

        update_connections(
            yt,
            self.Parameters.source_connections_directory,
            self.Parameters.destination_connections,
        )
