# coding: utf-8
import logging

import sandbox.sdk2 as sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common import task_env
from sandbox.projects.ydo import ydo_releasers
from sandbox.projects.ydo import execute_cmd, YdoFeaturesJson
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2
from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsDump


logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


class YdoComplaintsTransporterExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoComplaintsTransporter(sdk2.Task, SolomonMixinV2):
    """
    Создание инцидентов и тикетов арбитража для сервисной модели
    """

    class Requirements(task_env.TinyRequirements):
        ram = 1024
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    class Parameters(
        generate_view(
            ydo_complaints_transporter=YdoComplaintsTransporterExecutable, check_owner=True, with_environ=True
        )
    ):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)
        with sdk2.parameters.Group('YT_TABLES'):
            service_complaints_table = sdk2.parameters.String('Path to service complaints backup', required=True)
            service_incidents_table = sdk2.parameters.String('Path to service incidents backup', required=True)
        with sdk2.parameters.Group('SAMSARA'):
            oauth_token_name = sdk2.parameters.String('OAuth token name for robot-services', required=True)
            samsara_api_host = sdk2.parameters.String('Samsara host', required=True)

        complaints_transporter_config_resource = sdk2.parameters.String(
            'Complaints transporter config resource',
            required=True,
            choices=[('test', '/test_complaints_transporter_config'), ('prod', '/prod_complaints_transporter_config')],
        )
        create_incidents = sdk2.parameters.Bool('Create incidents from complaints', default=False)
        create_arbitrage_tickets = sdk2.parameters.Bool(
            'Create arbitrage samsara tickets from incidents', default=False
        )
        from_ts = sdk2.parameters.Integer('From timestamp to transport complaints', default=0)
        logging_config = sdk2.parameters.String('LoggingConfig')
        saas_orders_host = sdk2.parameters.String('Saas orders host', required=True)

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def on_execute(self):
        rubrics_json_prod = get_last_resource(YdoMergedRubricsDump, 'stable')
        features_json_prod = get_last_resource(YdoFeaturesJson, 'stable')

        cmd = [
            self.Parameters.ydo_complaints_transporter_path,
            '--kikimr-host',
            self.Parameters.ydb_host,
            '--kikimr-home',
            self.Parameters.ydb_home,
            '--kikimr-database',
            self.Parameters.ydb_database,
            '--kikimr-port',
            str(self.Parameters.ydb_port),
            '--samsara_api_host',
            self.Parameters.samsara_api_host,
            '--service_complaints',
            self.Parameters.service_complaints_table,
            '--service_incidents',
            self.Parameters.service_incidents_table,
            '--oauth_token_name',
            self.Parameters.oauth_token_name,
            '--complaints_transporter_config_resource',
            str(self.Parameters.complaints_transporter_config_resource),
            '--saas_orders_host',
            str(self.Parameters.saas_orders_host),
            '--rubrics_json',
            str(sdk2.ResourceData(rubrics_json_prod).path),
            '--features_json',
            str(sdk2.ResourceData(features_json_prod).path),
        ]
        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)
        self.add_param_to_cmd(cmd, '--kikimr-retry-count', str(self.Parameters.ydb_retry_count))
        self.add_param_to_cmd(cmd, '--kikimr-delay', str(self.Parameters.ydb_delay))
        self.add_param_to_cmd(cmd, '--kikimr-backoff', str(self.Parameters.ydb_backoff))
        self.add_param_to_cmd(cmd, '--kikimr-limit', str(self.Parameters.ydb_limit))

        if self.Parameters.create_incidents:
            cmd += ['--create_incidents']
        if self.Parameters.create_arbitrage_tickets:
            cmd += ['--create_arbitrage_tickets']
        if self.Parameters.from_ts:
            cmd += ['--from_ts', str(self.Parameters.from_ts)]

        execute_cmd(
            cmd, 'ydo_complaints_transporter', 'ydo_complaints_transporter failed', env=self.Parameters.get_environ()
        )

    def on_break(self, *args, **kwargs):
        SolomonMixinV2.on_break(self, *args, **kwargs)
        super(YdoComplaintsTransporter, self).on_break(*args, **kwargs)

    def on_finish(self, *args, **kwargs):
        SolomonMixinV2.on_finish(self, *args, **kwargs)
        super(YdoComplaintsTransporter, self).on_finish(*args, **kwargs)


def get_last_resource(resource_type, release_status):
    last_resource = resource_type.find(attrs=dict(released=release_status)).first()
    if last_resource is not None:
        return last_resource
    return None
