import os
import tempfile
import time
import uuid

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.kikimr.resources import YdbCliBinary
from sandbox.projects.ydo import execute_cmd


class YdoCopyTablesBetweenDatabases(sdk2.Task):
    class Parameters(sdk2.Parameters):
        ydb_cli_binary = sdk2.parameters.Resource(
            "YDB CLI Binary",
            resource_type=YdbCliBinary,
            required=True,
        )

        with sdk2.parameters.Group('[From] YDB config') as ydb_config_block_from:
            endpoint_from = sdk2.parameters.String('endpoint', required=True)
            database_from = sdk2.parameters.String('database', required=True)

        with sdk2.parameters.Group('[To] YDB config') as ydb_config_block_to:
            endpoint_to = sdk2.parameters.String('endpoint', required=True)
            database_to = sdk2.parameters.String('database', required=True)

        tables_to_copy = sdk2.parameters.List('Tables to copy', required=True, default=[])

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def copy_table(self, table_path, ydb_cli_path, env):
        table_path = os.path.normpath(table_path)
        table_name = os.path.basename(table_path)
        log_suffix = table_path.replace('/', '_')
        local_tmp_dir = tempfile.mkdtemp()
        ydb_tmp_dir = 'tmp/{}{}'.format(int(time.time()), uuid.uuid4())

        ydb_from = YdbCliHelper(
            ydb_cli_path, env, log_suffix, self.Parameters.endpoint_from, self.Parameters.database_from
        )
        ydb_to = YdbCliHelper(ydb_cli_path, env, log_suffix, self.Parameters.endpoint_to, self.Parameters.database_to)

        ydb_from.execute('tools', 'dump', '--path', table_path, '--output', local_tmp_dir)
        ydb_to.execute('scheme', 'mkdir', ydb_tmp_dir)
        ydb_to.execute('tools', 'restore', '--input', local_tmp_dir, '--path', ydb_tmp_dir)
        try:
            ydb_to.execute(
                'tools',
                'rename',
                '--item',
                'replace=True,source={},destination={}'.format(os.path.join(ydb_tmp_dir, table_name), table_path),
                fail_on_error=False,
            )
        except TaskError:
            self.set_info('exception during ydb_tools_rename_replace')
            ydb_to.execute(
                'tools',
                'rename',
                '--item',
                'replace=False,source={},destination={}'.format(os.path.join(ydb_tmp_dir, table_name), table_path),
            )
        ydb_to.execute('scheme', 'rmdir', ydb_tmp_dir)

    def on_execute(self):
        env = os.environ.copy()
        env['YDB_TOKEN'] = sdk2.Vault.data(self.owner, 'ydb-token')

        cmd_params = ['tar', '-xzf', str(sdk2.ResourceData(self.Parameters.ydb_cli_binary).path)]
        execute_cmd(cmd_params, 'extract_ydb_cli')
        ydb_cli_path = str(sdk2.path.Path.cwd() / 'ydb')

        for table_path in self.Parameters.tables_to_copy:
            self.copy_table(table_path, ydb_cli_path, env)


class YdbCliHelper(object):
    def __init__(self, ydb_cli_path, env, log_suffix, endpoint, database):
        self.ydb_cli_path = ydb_cli_path
        self.env = env
        self.log_suffix = log_suffix
        self.endpoint = endpoint
        self.database = database

    def execute(self, command, subcommand, *args, **kwargs):
        cmd_params = (
            [
                self.ydb_cli_path,
                '--verbose',
                '--endpoint',
                self.endpoint,
                '--database',
                self.database,
            ]
            + [command, subcommand]
            + list(args)
        )
        log_name = 'ydb_{}_{}_{}'.format(command, subcommand, self.log_suffix)
        execute_cmd(cmd_params, log_name, env=self.env, fail_on_error=kwargs.get('fail_on_error', True))
