import datetime

from sandbox import sdk2
from sandbox.projects.ydo import ydo_releasers

from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoCustomerAttributesGeneratorExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoCustomerAttributesGenerator(sdk2.Task, SolomonMixinV2):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(generate_view(ydo_customer_attributes_generator=YdoCustomerAttributesGeneratorExecutable, check_owner=True, with_environ=True)):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)
        with sdk2.parameters.Group('YT_TABLES'):
            yt_orders = sdk2.parameters.String('YT orders path', required=True)
            yt_antifraud_glues = sdk2.parameters.String('YT antifraud_glues path', default='')
            yt_customer_attributes = sdk2.parameters.String('YT customer_attributes path', default='')
        with sdk2.parameters.Group('BLACKBOX'):
            tvm_id = sdk2.parameters.Integer('TVM id')
            tvm_token_name = sdk2.parameters.String('TVM token env variable name')
            blackbox_env = sdk2.parameters.String('Blackbox env', choices=[('test', 'test'), ('mimino', 'mimino'), ('prod', 'prod')])

        logging_config = sdk2.parameters.String('LoggingConfig')
        from_date = sdk2.parameters.String('FromDate in a format "YYYY-mm-dd"')

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def on_execute(self):
        cmd = [
            self.Parameters.ydo_customer_attributes_generator_path,
            '--kikimr-host', self.Parameters.ydb_host,
            '--kikimr-home', self.Parameters.ydb_home,
            '--kikimr-database', self.Parameters.ydb_database,
            '--kikimr-port', str(self.Parameters.ydb_port),

            '--orders', self.Parameters.yt_orders,
        ]
        self.add_param_to_cmd(cmd, '--customer_attributes_backup', self.Parameters.yt_customer_attributes)
        self.add_param_to_cmd(cmd, '--antifraud_glues', self.Parameters.yt_antifraud_glues)

        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)
        self.add_param_to_cmd(cmd, '--kikimr-retry-count', str(self.Parameters.ydb_retry_count))
        self.add_param_to_cmd(cmd, '--kikimr-delay', str(self.Parameters.ydb_delay))
        self.add_param_to_cmd(cmd, '--kikimr-backoff', str(self.Parameters.ydb_backoff))
        self.add_param_to_cmd(cmd, '--kikimr-limit', str(self.Parameters.ydb_limit))
        self.add_param_to_cmd(cmd, '--from_date', self.Parameters.from_date or datetime.datetime.utcnow().strftime('%Y-%m-%d'))

        self.add_param_to_cmd(cmd, '--tvm_id', str(self.Parameters.tvm_id))
        self.add_param_to_cmd(cmd, '--tvm_token_name', str(self.Parameters.tvm_token_name))
        self.add_param_to_cmd(cmd, '--blackbox_env', str(self.Parameters.blackbox_env))

        execute_cmd(cmd, 'ydo_customer_attributes_generator', 'ydo_customer_attributes_generator failed', env=self.Parameters.get_environ())

    def on_break(self, *args, **kwargs):
        SolomonMixinV2.on_break(self, *args, **kwargs)
        super(YdoCustomerAttributesGenerator, self).on_break(*args, **kwargs)

    def on_finish(self, *args, **kwargs):
        SolomonMixinV2.on_finish(self, *args, **kwargs)
        super(YdoCustomerAttributesGenerator, self).on_finish(*args, **kwargs)
