from datetime import date, timedelta
import logging
import os

from sandbox import sdk2

from sandbox.projects.ydo import ydo_releasers, execute_cmd


# Resource types
class YdoDeleteUnactualWorkersExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoDeleteUnactualWorkers(sdk2.Task):
    """Task that deletes workers that were deleted from passport"""

    class Parameters(sdk2.Parameters):
        deleter = sdk2.parameters.Resource(
            'Executable',
            resource_type=YdoDeleteUnactualWorkersExecutable,
            required=True
        )

        with sdk2.parameters.Group("YDB parameters") as ydb_block:
            ydb_vault_token = sdk2.parameters.String("Your ydb token name in vault", default="ydb-token", required=True)
            ydb_host = sdk2.parameters.String("Host", required=True)
            ydb_port = sdk2.parameters.String("Port", required=True)
            ydb_database = sdk2.parameters.String("Database", required=True)
            ydb_home = sdk2.parameters.String("Home", required=True)

        yt_backup_dir = sdk2.parameters.String('Path to database backup on YT', required=True)
        max_delete_perc = sdk2.parameters.String('Max allowed percent for deleting (else fail)', default='0.01', required=True)
        saas_host = sdk2.parameters.String("SAAS host")

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def resolve(self):
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        env = os.environ.copy()
        env['YT_TOKEN'] = yt_token
        env['YT_PROXY'] = 'hahn.yt.yandex.net'
        env['YT_LOG_LEVEL'] = 'INFO'

        env['YDB_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.ydb_vault_token)

        logging.info('Running...')
        deleter = str(sdk2.ResourceData(self.Parameters.deleter).path)
        yesterday = (date.today() - timedelta(days=1)).strftime('%Y-%m-%d')
        cmd_params = [
            deleter,
            '--ydb_host', self.Parameters.ydb_host,
            '--ydb_port', self.Parameters.ydb_port,
            '--ydb_home', self.Parameters.ydb_home,
            '--ydb_database', self.Parameters.ydb_database,
            '--workers_path', os.path.join(self.Parameters.yt_backup_dir, 'workers'),
            '--max_delete_perc', self.Parameters.max_delete_perc,
            '--passport_userdata_path', '//statbox/heavy-dict/passport_userdata/' + yesterday,
        ]
        self.add_param_to_cmd(cmd_params, '--saas_host', self.Parameters.saas_host)

        execute_cmd(
            cmd_params,
            'accounts_deleter',
            'Failed to delete unactual accounts',
            env=env
        )

    def on_execute(self):
        self.resolve()
        logging.info('Done')
