import logging

from sandbox import sdk2

from sandbox.projects.ydo import ydo_releasers, execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view, ResourceWrapper
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE


# Resource types
class YdoResolveAddressesExecutable(sdk2.Resource):
    """
        addresses resolver
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoResolveAddresses(sdk2.Task):
    '''Task that resolves address and areas uris'''

    class Parameters(generate_view(
            resolver=YdoResolveAddressesExecutable,
            geobase_snapshot=ResourceWrapper(GEODATA6BIN_STABLE),
            check_owner=True,
            with_environ=True,
            default_secret_env_vars={'TVM_SECRET': 'ydo_tvm_secret_prod'})):

        with sdk2.parameters.Group("YDB parameters") as ydb_block:
            ydb_vault_token = sdk2.parameters.String("Your ydb token name in vault", default="ydb-token", required=True)
            ydb_host = sdk2.parameters.String("Host", required=True)
            ydb_port = sdk2.parameters.String("Port", required=True)
            ydb_database = sdk2.parameters.String("Database", required=True)
            ydb_home = sdk2.parameters.String("Home", required=True)

        yt_backup_dir = sdk2.parameters.String('Path to database backup on YT', required=True)

        regeocode_uri = sdk2.parameters.String('Regeocode uri')

        regeocode_all = sdk2.parameters.Bool('Regeocode all uris', default=False)

        actuality_period = sdk2.parameters.Integer('Regeocode uris older then (in seconds)', required=False)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def resolve(self):
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        env = self.Parameters.get_environ()
        env['YT_TOKEN'] = yt_token
        env['YT_PROXY'] = 'hahn.yt.yandex.net'
        env['YT_LOG_LEVEL'] = 'INFO'

        if self.Parameters.ydb_host != 'ydb-services.yandex.net':
            env['YDB_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.ydb_vault_token)

        logging.info('Resolving...')
        cmd_params = [
            self.Parameters.resolver_path,
            '--ydb_host', self.Parameters.ydb_host,
            '--ydb_port', self.Parameters.ydb_port,
            '--ydb_database', self.Parameters.ydb_database,
            '--ydb_home', self.Parameters.ydb_home,
            '--db_dump_dir', self.Parameters.yt_backup_dir,
            '--dump_from_yt',
            '--geobase_file', self.Parameters.geobase_snapshot_path,
            '--tvm_secret', env['TVM_SECRET'],
        ]
        if self.Parameters.regeocode_uri:
            with open('regeocode_uris_file', 'w') as regeocode_uris_file:
                regeocode_uris_file.write(self.Parameters.regeocode_uri)
            cmd_params += ['--regeocode', '--regeocode_uris_file', 'regeocode_uris_file']
        elif self.Parameters.regeocode_all:
            cmd_params += ['--regeocode', '--actuality_period', '0']
        elif self.Parameters.actuality_period is not None:
            cmd_params += ['--actuality_period', str(self.Parameters.actuality_period)]

        execute_cmd(
            cmd_params,
            'addresses_resolver',
            'Failed to resolve addresses',
            env=env
        )

    def on_execute(self):
        self.resolve()
        logging.info('Done')
