import logging
import time
import os.path

from sandbox import sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.ydo import ydo_releasers, execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view, ResourceWrapper
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2
from sandbox.projects.geosearch.CleanupYtFolder import clean_folder
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables


class YdoValidateProfilesExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoValidateProfiles(sdk2.Task, SolomonMixinV2):
    """Task that validates profiles in ydo db"""

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(
        generate_view(
            executable=YdoValidateProfilesExecutable,
            geobase_snapshot=ResourceWrapper(GEODATA6BIN_STABLE),
            with_environ=True,
        )
    ):
        with sdk2.parameters.Group("YDB parameters") as ydb_block:
            ydb_vault_token = sdk2.parameters.String("Your ydb token name in vault", default="ydb-token", required=True)
            ydb_host = sdk2.parameters.String("Host", required=True)
            ydb_port = sdk2.parameters.String("Port", required=True)
            ydb_database = sdk2.parameters.String("Database", required=True)
            ydb_home = sdk2.parameters.String("Home", required=True)
        with sdk2.parameters.Group("IntApi"):
            tvm_id = sdk2.parameters.Integer('TVM id')
            intapi_host = sdk2.parameters.String(
                'IntApi host',
                choices=[('mock', ''), ('test', 'shared-dev.hamster.yandex.ru'), ('prod', 'ydo-internal.yandex.net')],
            )

        yt_host = sdk2.parameters.String('YT host', default='hahn')
        yt_backup_dir = sdk2.parameters.String('Path to database backup on YT', required=True)
        yt_service_prices = sdk2.parameters.String('Service price stat table for price adequacy checking', default='')
        ogrn_whitelist = sdk2.parameters.String('Path to ogrn whitelist table (allow duplicates)', required=False)
        phones_whitelist = sdk2.parameters.String('Path to phones whitelist table (allow duplicates)', required=False)
        max_workers_to_change_status = sdk2.parameters.Integer('Max workers to change status by strikes')

        check_violations = sdk2.parameters.Bool('Check violations', default=True)
        check_strikes = sdk2.parameters.Bool('Check strikes', default=True)
        unpublish_passport_duplicates = sdk2.parameters.Bool(
            'Unpublish workers with passport duplicates', default=False
        )
        published_workers_out_dir = sdk2.parameters.String(
            'Output directory for published workers list', required=False
        )
        published_workers_out_table = sdk2.parameters.String(
            'Table name for published workers list. Joins with published_workers_out_dir', required=False
        )
        published_workers_create_link = sdk2.parameters.Bool(
            'Create link published_workers_out_dir/current -> published_workers_out_dir/published_workers_out_table',
            default=True,
        )
        published_workers_clean_dir = sdk2.parameters.Bool(
            'Remove old tables for published workers lists',
            default=True,
        )

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def link_published_workers_table(self, table_path, link_path):
        tables_to_link = {
            table_path: link_path,
        }

        link_task = YdoBackupLinkTables(
            self,
            description='Link tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=self.Parameters.yt_host,
            yt_vault_token='yt-token',
            yt_tables=tables_to_link,
        )
        link_task.enqueue()

        raise sdk2.WaitTask(
            [link_task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True
        )

    def on_execute(self):
        cmd = [
            self.Parameters.executable_path,
            '--ydb_host',
            self.Parameters.ydb_host,
            '--ydb_port',
            self.Parameters.ydb_port,
            '--ydb_database',
            self.Parameters.ydb_database,
            '--ydb_home',
            self.Parameters.ydb_home,
            '--db_dump_dir',
            self.Parameters.yt_backup_dir,
            '--dump_from_yt',
            '--geobase_file',
            self.Parameters.geobase_snapshot_path,
        ]
        self.add_param_to_cmd(cmd, '--service_prices', self.Parameters.yt_service_prices)
        self.add_param_to_cmd(cmd, '--tvm-id', str(self.Parameters.tvm_id) if self.Parameters.tvm_id else None)
        self.add_param_to_cmd(cmd, '--ydo-intapi-host', self.Parameters.intapi_host)
        self.add_param_to_cmd(cmd, '--ogrn_whitelist', self.Parameters.ogrn_whitelist)
        self.add_param_to_cmd(cmd, '--phones_whitelist', self.Parameters.phones_whitelist)
        self.add_param_to_cmd(
            cmd,
            '--max_workers_to_change_status',
            str(self.Parameters.max_workers_to_change_status) if self.Parameters.max_workers_to_change_status else None,
        )

        published_workers_out_path = None
        published_workers_out_dir = self.Parameters.published_workers_out_dir
        if published_workers_out_dir:
            if self.Parameters.published_workers_out_table:
                published_workers_out_path = os.path.join(
                    published_workers_out_dir, self.Parameters.published_workers_out_table
                )
            else:
                current_timestamp = int(time.time())
                published_workers_out_path = os.path.join(published_workers_out_dir, str(current_timestamp))

            self.add_param_to_cmd(cmd, '--published_workers_out_path', published_workers_out_path)

        if self.Parameters.unpublish_passport_duplicates:
            cmd += ['--unpublish_passport_duplicates']
        if self.Parameters.check_violations:
            cmd += ['--check_violations']
        if self.Parameters.check_strikes:
            cmd += ['--check_strikes']

        with self.memoize_stage.execute_cmd:
            execute_cmd(cmd, 'validate_profiles', 'Failed to validate profiles', env=self.Parameters.get_environ())

        if self.Parameters.check_strikes and published_workers_out_dir:
            if self.Parameters.published_workers_create_link:
                with self.memoize_stage.create_link_to_current:
                    self.link_published_workers_table(
                        link_path=os.path.join(published_workers_out_dir, 'current'),
                        table_path=published_workers_out_path,
                    )

            if self.Parameters.published_workers_clean_dir:
                with self.memoize_stage.clean_folder:
                    clean_folder(
                        self,
                        published_workers_out_dir,
                        yt_host=self.Parameters.yt_host,
                        history_size=85,
                    )

    def on_break(self, *args, **kwargs):
        SolomonMixinV2.on_break(self, *args, **kwargs)
        super(YdoValidateProfiles, self).on_break(*args, **kwargs)

    def on_finish(self, *args, **kwargs):
        SolomonMixinV2.on_finish(self, *args, **kwargs)
        super(YdoValidateProfiles, self).on_finish(*args, **kwargs)
