import datetime

from sandbox import sdk2
from sandbox.projects.ydo import execute_cmd, ydo_releasers
from sandbox.projects.ydo.parameters_view import generate_view_for_yt


class YdoDbLogsSolomonAccumulatorBinary(sdk2.Resource):
    releasers = ydo_releasers
    executable = True
    releasable = True


class YdoAccumulateDbLogsForSolomon(sdk2.Task):
    """ Accumulate DB logs from YT to Solomon. """

    class Parameters(generate_view_for_yt(
        ydo_db_logs_solomon_accumulator_binary=YdoDbLogsSolomonAccumulatorBinary,
        check_owner=True,
        with_environ=True,
        default_secret_env_vars={
            'YT_TOKEN': 'yt-token',
            'YQL_TOKEN': 'YQL_TOKEN',
            'SOLOMON_TOKEN': 'solomon-token'
        },
    )):
        input_date = sdk2.parameters.RadioGroup(
            'Input date',
            choices=(
                ('yesterday', 'yesterday'),
                ('custom', 'custom'),
            ),
            required=True,
            default='yesterday',
            sub_fields={'custom': ['custom_input_date']}
        )
        custom_input_date = sdk2.parameters.StrictString(
            'YYYY-MM-DD',
            regexp=r'\d{4}-\d{2}-\d{2}',
            required=True
        )
        solomon_project = sdk2.parameters.String(
            'Solomon project',
            required=True,
            default='ydo-db-logs'
        )
        solomon_cluster = sdk2.parameters.String(
            'Solomon cluster',
            required=True,
            default='prestable'
        )
        solomon_service = sdk2.parameters.String(
            'Solomon service',
            required=True,
            default='db_logs'
        )

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_input_date(self):
        if self.Parameters.input_date == 'yesterday':
            yesterday = datetime.datetime.today() - datetime.timedelta(days=1)
            input_date = yesterday.strftime('%Y-%m-%d')
        elif self.Parameters.input_date == 'custom':
            input_date = self.Parameters.custom_input_date
        return input_date

    def on_execute(self):
        input_date = self.get_input_date()
        cmd = [
            self.Parameters.ydo_db_logs_solomon_accumulator_binary_path,
            '--input_date', input_date,
            '--project', self.Parameters.solomon_project,
            '--cluster', self.Parameters.solomon_cluster,
            '--service', self.Parameters.solomon_service,
        ]

        execute_cmd(
            cmd=cmd,
            log_name='ydo_db_log_solomon_accumulator',
            error_message='Database logs accumulator failed',
            env=self.Parameters.get_environ(),
        )
