# -*- coding: utf-8 -*-
import datetime
import logging
import os
import tempfile

import sandbox.common.types.task as ctt
from sandbox import sdk2

from sandbox.projects.ydo import (
    ydo_releasers,
    execute_cmd,
    YdoFeaturesJson,
)

from sandbox.projects.common.geosearch.utils import unpack_file

from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE
from sandbox.projects.ydo.resource_types import (
    YdoDispatchCacheIndexerExecutable,
    YdoDispatchCacheIndexerConfigs,
    YdoDispatchShooterConfigs,
    YdoDispatchCacheReferencePoints,
    YdoDispatchCachePartnerFlowServices,
    YdoServiceModelCategoriesJson
)

from sandbox.sandboxsdk import environments


class YdoDispatchCacheIndexer(sdk2.Task):
    """Task that creates dispatch cache"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400

        with sdk2.parameters.Group("Required resources") as required_resources_block:
            dispatch_cache_indexer_binary = sdk2.parameters.Resource(
                "dispatch cache indexer executable",
                resource_type=YdoDispatchCacheIndexerExecutable,
                required=True,
            )
            dispatch_cache_indexer_config = sdk2.parameters.Resource(
                "dispatch cache indexer config",
                resource_type=YdoDispatchCacheIndexerConfigs,
                required=True,
            )
            dispatch_shooter_config = sdk2.parameters.Resource(
                "dispatch shooter config",
                resource_type=YdoDispatchShooterConfigs,
                required=True,
            )
            dispatch_cache_reference_points = sdk2.parameters.Resource(
                "dispatch cache reference points",
                resource_type=YdoDispatchCacheReferencePoints,
                required=True
            )
            dispatch_cache_partner_flow_services = sdk2.parameters.Resource(
                "dispatch cache partner flow service list",
                resource_type=YdoDispatchCachePartnerFlowServices
            )
            geobase_snapshot = sdk2.parameters.Resource(
                'Geobase 6 snapshot',
                resource_type=GEODATA6BIN_STABLE,
                required=True,
            )
            geodata_tz_snapshot = sdk2.parameters.Resource(
                'Geodata tz',
                resource_type=GEODATATZDATA_STABLE,
                required=True,
            )
            rubrics_json = sdk2.parameters.Resource(
                'merged rubrics.json',
                resource_type=YdoMergedRubricsSmallDump,
                required=True,
            )
            features_json = sdk2.parameters.Resource(
                'Features',
                resource_type=YdoFeaturesJson,
                required=True,
            )
            service_model_categories_json = sdk2.parameters.Resource(
                'service model categories json',
                resource_type=YdoServiceModelCategoriesJson,
                required=True,
            )

        with sdk2.parameters.Group("Environment") as env_block:
            with sdk2.parameters.RadioGroup('Ydo ctype name') as ctype:
                ctype.values['hamster'] = ctype.Value('hamster')
                ctype.values['testing'] = ctype.Value('testing')

            ydb_token = sdk2.parameters.YavSecret("Ydb token in yav", required=True)
            tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)
            saas_kv_lb_tvm_token = sdk2.parameters.YavSecret("Tvm sercet for saas_kv logbroker", required=True)

        with sdk2.parameters.Group("Miscellaneous parameters") as miscellaneous_block:
            yt_log_level = sdk2.parameters.String("Yt log level", required=False)
            kps = sdk2.parameters.Integer("saas_kv kps", required=False)
            slots_search_interval_days = sdk2.parameters.Integer("interval [today, today+X] for slots", required=False)
            shoot_workers_number = sdk2.parameters.Integer("number of shooter workers", required=False)
            upload_workers_number = sdk2.parameters.Integer("number of upload workers", required=False)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment("yandex-yt"),
        ]
        cores = 1
        disk_space = 5 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def run_cache_indexer(self):
        yt_token = sdk2.Vault.data(self.owner, "yt-token")

        env = os.environ.copy()
        env["YT_TOKEN"] = yt_token
        env["YT_LOG_LEVEL"] = self.Parameters.yt_log_level if self.Parameters.yt_log_level else "INFO"
        env["TVM_SECRET"] = self.Parameters.tvm_secret.data()["TVM_SECRET"]
        env["YDB_TOKEN"] = self.Parameters.ydb_token.data()["YDB_TOKEN"]
        env["SAAS_LOGBROKER_TVM_SECRET"] = self.Parameters.saas_kv_lb_tvm_token.data()["TVM_SECRET"]

        tmp_dir = tempfile.mkdtemp()
        unpack_file(str(sdk2.ResourceData(self.Parameters.geodata_tz_snapshot).path), tmp_dir)
        os.symlink(str(sdk2.ResourceData(self.Parameters.geobase_snapshot).path), os.path.join(tmp_dir, "geodata6.bin"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.rubrics_json).path), os.path.join(tmp_dir, "rubrics.json"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.features_json).path), os.path.join(tmp_dir, "features.json"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.service_model_categories_json).path), os.path.join(tmp_dir, "service_model_categories.json"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.dispatch_cache_reference_points).path), os.path.join(tmp_dir, "reference_points.json"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.dispatch_cache_partner_flow_services).path), os.path.join(tmp_dir, "partner_flow_services.json"))

        logging.info("Shooting...")
        cmd = [
            str(sdk2.ResourceData(self.Parameters.dispatch_cache_indexer_binary).path),
            "--config", os.path.join(str(sdk2.ResourceData(self.Parameters.dispatch_cache_indexer_config).path), str(self.Parameters.ctype) + ".yaml"),
            "--dispatch_shooter_config", os.path.join(str(sdk2.ResourceData(self.Parameters.dispatch_shooter_config).path), str(self.Parameters.ctype) + ".yaml"),
            "--reference_geoids", os.path.join(tmp_dir, "reference_points.json"),
            "--partner_flow_services", os.path.join(tmp_dir, "partner_flow_services.json"),
        ]

        if self.Parameters.slots_search_interval_days:
            cmd.extend([
                "--slots_search_interval_days", self.Parameters.slots_search_interval_days
            ])

        if self.Parameters.kps:
            cmd.extend([
                "--kps", self.Parameters.kps
            ])

        if self.Parameters.shoot_workers_number:
            cmd.extend([
                "--shoot_workers_number", self.Parameters.shoot_workers_number
            ])

        if self.Parameters.upload_workers_number:
            cmd.extend([
                "--upload_workers_number", self.Parameters.upload_workers_number
            ])

        cmd = [str(arg) for arg in cmd]
        execute_cmd(
            cmd,
            "ydo_dispatch_cache_indexer",
            "Failed to run indexer binary",
            env=env,
            cwd=tmp_dir,
        )

    def on_execute(self):

        with self.memoize_stage.run_cache_indexer:
            self.run_cache_indexer()

        logging.info("Done")
