# -*- coding: utf-8 -*-
import datetime
import logging
import os
import tempfile

import sandbox.common.types.task as ctt
from sandbox import sdk2

from sandbox.projects.ydo import (
    ydo_releasers,
    execute_cmd,
    YdoFeaturesJson,
)

from sandbox.projects.common.geosearch.utils import unpack_file

from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE
from sandbox.projects.ydo.resource_types import (
    YdoDispatchShooterExecutable,
    YdoDispatchShooterConfigs,
    YdoServiceModelCategoriesJson,
)

from sandbox.sandboxsdk import environments


class YdoDispatchShooter(sdk2.Task):
    """Task that shooting to dispatch and erp virtual orders and save slots to yt"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400

        with sdk2.parameters.Group("Required resources") as required_resources_block:
            dispatch_shooter = sdk2.parameters.Resource(
                "dispatch shooter executable",
                resource_type=YdoDispatchShooterExecutable,
                required=True,
            )
            dispatch_shooter_config = sdk2.parameters.Resource(
                "dispatch shooter config",
                resource_type=YdoDispatchShooterConfigs,
                required=True,
            )
            geobase_snapshot = sdk2.parameters.Resource(
                'Geobase 6 snapshot',
                resource_type=GEODATA6BIN_STABLE,
                required=True,
            )
            geodata_tz_snapshot = sdk2.parameters.Resource(
                'Geodata tz',
                resource_type=GEODATATZDATA_STABLE,
                required=True,
            )
            rubrics_json = sdk2.parameters.Resource(
                'merged rubrics.json',
                resource_type=YdoMergedRubricsSmallDump,
                required=True,
            )
            features_json = sdk2.parameters.Resource(
                'Features',
                resource_type=YdoFeaturesJson,
                required=True,
            )
            service_model_categories_json = sdk2.parameters.Resource(
                'service model categories json',
                resource_type=YdoServiceModelCategoriesJson,
                required=True,
            )


        with sdk2.parameters.Group("Environment") as env_block:
            with sdk2.parameters.RadioGroup('Ydo ctype name') as ctype:
                ctype.values['hamster'] = ctype.Value('hamster')
                ctype.values['testing'] = ctype.Value('testing')

            ydb_token = sdk2.parameters.YavSecret("Ydb token in yav", required=True)
            tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)

        with sdk2.parameters.Group("Input tables") as required_resources_block:
            virtual_orders_table = sdk2.parameters.String("Path to table with virtual orders for dispatch", required=True)

        with sdk2.parameters.Group("Output parameters") as output_block:
            res_dir = sdk2.parameters.String("Shooting resulting directory", required=True)

        with sdk2.parameters.Group("Miscellaneous parameters") as miscellaneous_block:
            yt_log_level = sdk2.parameters.String("Yt log level", required=False)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment("yandex-yt"),
        ]
        cores = 1
        disk_space = 5 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def shoot_dispatch(self):
        yt_token = sdk2.Vault.data(self.owner, "yt-token")

        env = os.environ.copy()
        env["YT_TOKEN"] = yt_token
        env["YT_PROXY"] = "{}.yt.yandex.net".format(self.yt_host)
        env["YT_LOG_LEVEL"] = self.Parameters.yt_log_level if self.Parameters.yt_log_level else "INFO"
        env["TVM_SECRET"] = self.Parameters.tvm_secret.data()["TVM_SECRET"]
        env["YDB_TOKEN"] = self.Parameters.ydb_token.data()["YDB_TOKEN"]

        self.Context.shoot_day = str(datetime.datetime.utcnow().replace(microsecond=0).isoformat()) + '+00:00'
        self.Context.res_table = os.path.join(self.Parameters.res_dir, self.Context.shoot_day)

        tmp_dir = tempfile.mkdtemp()
        unpack_file(str(sdk2.ResourceData(self.Parameters.geodata_tz_snapshot).path), tmp_dir)
        os.symlink(str(sdk2.ResourceData(self.Parameters.geobase_snapshot).path), os.path.join(tmp_dir, "geodata6.bin"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.rubrics_json).path), os.path.join(tmp_dir, "rubrics.json"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.features_json).path), os.path.join(tmp_dir, "features.json"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.service_model_categories_json).path), os.path.join(tmp_dir, "service_model_categories.json"))

        logging.info("Shooting...")
        cmd = [
            str(sdk2.ResourceData(self.Parameters.dispatch_shooter).path),
            "--config", os.path.join(str(sdk2.ResourceData(self.Parameters.dispatch_shooter_config).path), str(self.Parameters.ctype) + ".yaml"),
            "--read_proxy", "{}.yt.yandex.net".format(self.yt_host),
            "--write_proxy", "{}.yt.yandex.net".format(self.yt_host),
            "--virtual_orders_table", self.Parameters.virtual_orders_table,
            "--res_table", self.Context.res_table,
        ]

        execute_cmd(
            cmd,
            "ydo_dispatch_shooter",
            "Failed to shoot dispatch",
            env=env,
            cwd=tmp_dir,
        )

    def on_execute(self):
        self.yt_host = "hahn"

        with self.memoize_stage.shoot_dispatch:
            self.shoot_dispatch()

        logging.info("Done")
