# -*- coding: utf-8 -*-
import datetime
import logging
import os
import tempfile

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
)
from sandbox.projects.geosearch.CleanupYtFolder import clean_folder
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables
from sandbox.projects.ydo.resource_types import (
    YdoOrdersPredictionExecutable,
)


class YdoOrdersPrediction(sdk2.Task):
    """Task calculate orders prediction"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400

        with sdk2.parameters.Group("Required resources") as required_resources_block:
            executable = sdk2.parameters.Resource(
                "orders prediction executable",
                resource_type=YdoOrdersPredictionExecutable,
                required=True,
            )
        history_table = sdk2.parameters.String("Path to history table", required=False)
        result_dir = sdk2.parameters.String("Path to result dir", required=False)
        result_datalens_dir = sdk2.parameters.String("Path to result datalens dir", required=False)

    def link(self):
        link_target = os.path.join(self.Parameters.result_dir, 'current')
        tables_to_link = {self.Context.res_table: link_target}
        link_task = YdoBackupLinkTables(
            self,
            description='Link tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=self.yt_host,
            yt_vault_token=self.Context.yt_vault_token,
            yt_tables=tables_to_link
        )
        link_task.enqueue()

        raise sdk2.WaitTask([link_task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True)

    def run(self):
        self.Context.yt_vault_token = "yt-token"
        yt_token = sdk2.Vault.data(self.owner, self.Context.yt_vault_token)

        env = os.environ.copy()
        env["YT_TOKEN"] = yt_token
        env["YT_PROXY"] = "{}.yt.yandex.net".format(self.yt_host)

        self.Context.day = str(datetime.datetime.utcnow().replace(microsecond=0).isoformat()) + '+00:00'
        self.Context.res_table = os.path.join(self.Parameters.result_dir, self.Context.day)

        cmd = [
            str(sdk2.ResourceData(self.Parameters.executable).path),
            "--read_proxy", "{}.yt.yandex.net".format(self.yt_host),
            "--write_proxy", "{}.yt.yandex.net".format(self.yt_host),
            "--history_table", self.Parameters.history_table,
            "--result_table", self.Context.res_table,
            "--result_datalens_dir", self.Parameters.result_datalens_dir,
        ]

        tmp_dir = tempfile.mkdtemp()
        execute_cmd(
            cmd,
            "orders_prediction",
            "Failed to calculate prediction",
            env=env,
            cwd=tmp_dir,
        )

    def on_execute(self):
        self.yt_host = "hahn"

        with self.memoize_stage.running:
            self.run()

        with self.memoize_stage.clean_folder:
            clean_folder(self, self.Parameters.result_dir, yt_host=self.yt_host, yt_vault_token=self.Context.yt_vault_token)

        with self.memoize_stage.link_to_current:
            self.link()

        logging.info("Done")
