# -*- coding: utf-8 -*-
import logging
import os
import tempfile

from sandbox import sdk2
from sandbox.projects.common.geosearch.utils import unpack_file
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE

from sandbox.projects.ydo import (
    execute_cmd,
    YdoFeaturesJson,
    ydo_releasers
)
from sandbox.projects.ydo.default_multislot_task import YdoMutlihostTask
from sandbox.projects.ydo.parameters_view import generate_view, with_config, ResourceWrapper
from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoRedispatchOrdersForUndeterminedWorkersExecutable(sdk2.Resource):
    executable = True
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = 'ydo/tools/order/redispatch_orders_for_undetermined_workers'


class YdoRedispatchOrdersForUndeterminedWorkersTask(YdoMutlihostTask, SolomonMixinV2):
    """
        Task that redispatches order if assigned worker should, but didn't confirm his work for tomorrow.
        It gets orders from saas, filters them and creates async-api tasks for each of the filtered order.
    """
    class Parameters(with_config(
            generate_view(
                executable=YdoRedispatchOrdersForUndeterminedWorkersExecutable,
                geobase_snapshot=ResourceWrapper(GEODATA6BIN_STABLE),
                geodata_tz_snapshot=ResourceWrapper(GEODATATZDATA_STABLE),
                rubrics=YdoMergedRubricsSmallDump,
                features=YdoFeaturesJson,
                check_owner=True,
                with_environ=True,
                default_secret_env_vars={
                    'TVM_SECRET': 'tvm_secret',
                    'LOGBROKER_TOKEN': 'LOGBROKER_TOKEN',
                    'YDB_TOKEN': 'yql-token',
                }
            ),
            'ydo/tools/order/redispatch_orders_for_undetermined_workers/configs',
            [
                ('testing', 'testing.yaml'),
                ('production', 'production.yaml'),
            ]
    )):
        with sdk2.parameters.Group("Parameters"):
            logging_level = sdk2.parameters.Integer('Logging level', required=False, default=logging.DEBUG)

    def on_execute(self):
        logging.info("Redispatching...")

        tmp_dir = tempfile.mkdtemp()
        unpack_file(self.Parameters.geodata_tz_snapshot_path, str(tmp_dir))
        os.symlink(self.Parameters.geobase_snapshot_path, os.path.join(tmp_dir, "geodata6.bin"))
        os.symlink(self.Parameters.rubrics_path, os.path.join(tmp_dir, "rubrics.json"))
        os.symlink(self.Parameters.features_path, os.path.join(tmp_dir, "features.json"))

        cmd = [self.Parameters.executable_path, '--config', self.Parameters.config_path, '--logging_level', str(self.Parameters.logging_level)]
        execute_cmd(
            cmd, 'redispatch_orders_for_undetermined_workers',
            'redispatch_orders_for_undetermined_workers failed',
            env=self.Parameters.get_environ(),
            cwd=tmp_dir
        )
        logging.info("Done")

    def on_break(self, *args, **kwargs):
        SolomonMixinV2.on_break(self, *args, **kwargs)
        sdk2.Task.on_break(self, *args, **kwargs)

    def on_finish(self, *args, **kwargs):
        SolomonMixinV2.on_finish(self, *args, **kwargs)
        sdk2.Task.on_finish(self, *args, **kwargs)
