# -*- coding: utf-8 -*-
import datetime
import logging
import os
import tempfile

import sandbox.common.types.task as ctt
from sandbox import sdk2

from sandbox.projects.ydo import (
    ydo_releasers,
    execute_cmd,
    YdoFeaturesJson,
)

from sandbox.projects.common.geosearch.utils import unpack_file

from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE
from sandbox.projects.ydo.resource_types import (
    YdoDispatchDistanceGrid,
    YdoDispatchHomeGeopoints,
    YdoDispatchPolygonRestrictionsJson,
    YdoRedispatcherExecutable,
    YdoRedispatcherConfigs,
    YdoRedispatcherExactConfig,
    YdoServiceModelCategoriesJson,
    YdoDispatchLightPartnerConfig,
    YdoRedispatcherTransitionMap,
)


class YdoRedispatcher(sdk2.Task):
    """Task that redispatches orders from organization to workers"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400
        with sdk2.parameters.Group("Redispatch parameters") as redispatch_parameters:
            redispatch_date = sdk2.parameters.String(
                "redispatch date",
                required=False,
                default=(datetime.datetime.utcnow() + datetime.timedelta(days=1)).date().isoformat(),
            )
            organization_id = sdk2.parameters.String(
                "organization id",
                required=False,
            )
            no_switch = sdk2.parameters.Bool(
                "disable schedule switch",
                required=False,
                default=False,
            )
            exact_matching = sdk2.parameters.Bool(
                "use exact-matching strategy",
                required=False,
                default=False,
            )
            allow_virtuals = sdk2.parameters.List(
                "virtuals whitelist (space separated)",
                value_type=sdk2.parameters.String,
                required=False,
            )
            remove_virtuals = sdk2.parameters.List(
                "virtuals blacklist (space separated)",
                value_type=sdk2.parameters.String,
                required=False,
            )
            remove_older_ts = sdk2.parameters.Integer(
                "remove orders created before ts",
                required=False,
            )
            remove_specializations = sdk2.parameters.List(
                "specialization blacklist (space separated)",
                value_type=sdk2.parameters.String,
                required=False,
            )
            remove_orders = sdk2.parameters.List(
                "order ids blacklist (space separated)",
                value_type=sdk2.parameters.String,
                required=False,
            )
            rent_organization_ids = sdk2.parameters.List(
                "rent scheme organizations list",
                value_type=sdk2.parameters.String,
                required=False,
            )
            base_organization_ids = sdk2.parameters.List(
                "not-exact matching organizations list",
                value_type=sdk2.parameters.String,
                required=False,
            )
            transition_penalty_organization_ids = sdk2.parameters.List(
                "transition penalty organizations list",
                value_type=sdk2.parameters.String,
                required=False,
            )
            patch_filters_organization_ids = sdk2.parameters.List(
                "patch filters organizations list",
                value_type=sdk2.parameters.String,
                required=False,
            )
            draw_transition_map = sdk2.parameters.Bool(
                "draw transition map",
                required=False,
                default=False,
            )
            failed_orders_table = sdk2.parameters.String(
                "failed orders table",
                required=False,
            )
            request_orders = sdk2.parameters.Bool(
                "automatically request orders from erp",
                required=False,
                default=False,
            )
        with sdk2.parameters.Group("Required resources") as required_resources_block:
            redispatcher = sdk2.parameters.Resource(
                "redispatcher executable",
                resource_type=YdoRedispatcherExecutable,
                required=True,
            )
            redispatcher_config = sdk2.parameters.Resource(
                "redispatcher config",
                resource_type=YdoRedispatcherConfigs,
                required=True,
            )
            geobase_snapshot = sdk2.parameters.Resource(
                'Geobase 6 snapshot',
                resource_type=GEODATA6BIN_STABLE,
                required=True,
            )
            geodata_tz_snapshot = sdk2.parameters.Resource(
                'Geodata tz',
                resource_type=GEODATATZDATA_STABLE,
                required=True,
            )
            rubrics_json = sdk2.parameters.Resource(
                'merged rubrics.json',
                resource_type=YdoMergedRubricsSmallDump,
                required=True,
            )
            features_json = sdk2.parameters.Resource(
                'Features',
                resource_type=YdoFeaturesJson,
                required=True,
            )
            service_model_categories_json = sdk2.parameters.Resource(
                'service model categories json',
                resource_type=YdoServiceModelCategoriesJson,
                required=True,
            )
            home_geopoints_json = sdk2.parameters.Resource(
                'home geopoints json',
                resource_type=YdoDispatchHomeGeopoints,
                required=True,
            )
            polygon_restrictions_json = sdk2.parameters.Resource(
                'polygon restrictions json',
                resource_type=YdoDispatchPolygonRestrictionsJson,
                required=True,
            )
            distance_grid = sdk2.parameters.Resource(
                'distance grid',
                resource_type=YdoDispatchDistanceGrid,
                required=True,
            )
            light_partner_config = sdk2.parameters.Resource(
                "light partner config",
                resource_type=YdoDispatchLightPartnerConfig,
                required=True,
            )
            redispatcher_exact_config = sdk2.parameters.Resource(
                "redispatcher exact config",
                resource_type=YdoRedispatcherExactConfig,
                required=True,
            )

        with sdk2.parameters.Group("Environment") as env_block:
            with sdk2.parameters.RadioGroup('Ydo ctype name') as ctype:
                ctype.values['hamster'] = ctype.Value('hamster')
                ctype.values['testing'] = ctype.Value('testing')
                ctype.values['production'] = ctype.Value('production')

            ydb_token = sdk2.parameters.YavSecret("Ydb token in yav", required=True)
            tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)
            solomon_token = sdk2.parameters.YavSecret("Solomon token in yav", required=True)
            yt_token = sdk2.parameters.YavSecret("Yt token in yav", required=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 40 * 1024
        disk_space = 20 * 1024
        semaphores = ctt.Semaphores(
            acquires=[ctt.Semaphores.Acquire(name='ydo_redispatcher_queue')],
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def redispatch(self):
        env = os.environ.copy()
        env["TVM_SECRET"] = self.Parameters.tvm_secret.data()["TVM_SECRET"]
        env["YDB_TOKEN"] = self.Parameters.ydb_token.data()["YDB_TOKEN"]
        env["SOLOMON_TOKEN"] = self.Parameters.solomon_token.data()["SOLOMON_TOKEN"]
        env["YT_TOKEN"] = self.Parameters.yt_token.data()["YT_TOKEN"]

        tmp_dir = tempfile.mkdtemp()
        unpack_file(str(sdk2.ResourceData(self.Parameters.geodata_tz_snapshot).path), tmp_dir)
        os.symlink(str(sdk2.ResourceData(self.Parameters.geobase_snapshot).path), os.path.join(tmp_dir, "geodata6.bin"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.rubrics_json).path), os.path.join(tmp_dir, "rubrics.json"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.features_json).path), os.path.join(tmp_dir, "features.json"))
        os.symlink(
            str(sdk2.ResourceData(self.Parameters.service_model_categories_json).path),
            os.path.join(tmp_dir, "service_model_categories.json"),
        )
        os.symlink(
            str(sdk2.ResourceData(self.Parameters.home_geopoints_json).path),
            os.path.join(tmp_dir, "home_geopoints.json"),
        )
        os.symlink(
            str(sdk2.ResourceData(self.Parameters.polygon_restrictions_json).path),
            os.path.join(tmp_dir, "polygon_restrictions.json"),
        )
        os.symlink(
            str(sdk2.ResourceData(self.Parameters.distance_grid).path), os.path.join(tmp_dir, "distance_grid.tsv")
        )
        os.symlink(
            str(sdk2.ResourceData(self.Parameters.light_partner_config).path),
            os.path.join(tmp_dir, "light_partner_config.json"),
        )
        os.symlink(
            str(sdk2.ResourceData(self.Parameters.redispatcher_exact_config).path),
            os.path.join(tmp_dir, "redispatcher_exact_config.json"),
        )

        logging.info("Redispatching...")
        cmd = [
            str(sdk2.ResourceData(self.Parameters.redispatcher).path),
            "--config",
            os.path.join(
                str(sdk2.ResourceData(self.Parameters.redispatcher_config).path), str(self.Parameters.ctype) + ".yaml"
            ),
        ]
        if self.Parameters.no_switch:
            cmd.append("--no_switch")
        if self.Parameters.redispatch_date:
            cmd.extend(["--date", self.Parameters.redispatch_date])
        if self.Parameters.organization_id:
            cmd.extend(["--organization_id", self.Parameters.organization_id])
        if self.Parameters.exact_matching:
            cmd.append("--exact_matching")
        if self.Parameters.allow_virtuals:
            cmd.extend(["--allow_virtuals"] + self.Parameters.allow_virtuals)
        if self.Parameters.remove_virtuals:
            cmd.extend(["--remove_virtuals"] + self.Parameters.remove_virtuals)
        if self.Parameters.remove_older_ts:
            cmd.extend(["--remove_older_ts", str(self.Parameters.remove_older_ts)])
        if self.Parameters.remove_specializations:
            cmd.extend(["--remove_specializations"] + self.Parameters.remove_specializations)
        if self.Parameters.remove_orders:
            cmd.extend(["--remove_orders"] + self.Parameters.remove_orders)
        if self.Parameters.rent_organization_ids:
            cmd.extend(["--rent_organization_ids"] + self.Parameters.rent_organization_ids)
        if self.Parameters.base_organization_ids:
            cmd.extend(["--base_organization_ids"] + self.Parameters.base_organization_ids)
        if self.Parameters.transition_penalty_organization_ids:
            cmd.extend(["--transition_penalty_organization_ids"] + self.Parameters.transition_penalty_organization_ids)
        if self.Parameters.patch_filters_organization_ids:
            cmd.extend(["--patch_filters_organization_ids"] + self.Parameters.patch_filters_organization_ids)
        if self.Parameters.request_orders:
            cmd.append("--request_orders")
        if self.Parameters.failed_orders_table:
            cmd.extend(["--failed_orders_table", self.Parameters.failed_orders_table])

        if self.Parameters.draw_transition_map:
            resource = YdoRedispatcherTransitionMap(self, "Redispatcher transition map", "transition_map.html")
            resource.organization_id = self.Parameters.organization_id
            resource.date = self.Parameters.redispatch_date
            resource_data = sdk2.ResourceData(resource)

            cmd.extend(["--transition_map_path", str(resource_data.path)])

        execute_cmd(
            cmd,
            "ydo_redispatcher",
            "Failed to redispatch",
            env=env,
            cwd=tmp_dir,
        )

        if self.Parameters.draw_transition_map:
            resource_data.ready()

    def on_execute(self):
        with self.memoize_stage.redispatch:
            self.redispatch()

        logging.info("Done")
