# -*- coding: utf-8 -*-
import datetime
import logging
import os
import tempfile

from sandbox import sdk2

from sandbox.projects.ydo import (
    execute_cmd,
)

from sandbox.projects.ydo.resource_types import (
    YdoRedispatcherMonitoringExecutable,
    YdoRedispatcherConfigs,
    YdoDispatchLightPartnerConfig,
    YdoDispatchPolygonRestrictionsJson
)


class YdoRedispatcherMonitoring(sdk2.Task):
    """Task that redispatches orders from organization to workers"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400

        with sdk2.parameters.Group("Monitoring parameters") as monitoring_parameters:
            redispatch_date = sdk2.parameters.String(
                "redispatch date",
                required=False,
                default=(datetime.datetime.utcnow() + datetime.timedelta(days=1)).date().isoformat()
            )
            trigger_hour = sdk2.parameters.Integer(
                "fails start hour",
                required=False,
                default=16
            )
            organization_ids = sdk2.parameters.List(
                "monitoring organizations list",
                value_type=sdk2.parameters.String,
                required=True,
            )
            safe_organization_ids = sdk2.parameters.List(
                "monitoring organizations list with disabled alerts",
                value_type=sdk2.parameters.String,
                required=False,
            )

        with sdk2.parameters.Group("Required resources") as required_resources_block:
            redispatcher_monitoring = sdk2.parameters.Resource(
                "redispatcher monitoring executable",
                resource_type=YdoRedispatcherMonitoringExecutable,
                required=True,
            )
            redispatcher_config = sdk2.parameters.Resource(
                "redispatcher config",
                resource_type=YdoRedispatcherConfigs,
                required=True,
            )
            light_partner_config = sdk2.parameters.Resource(
                "light partner config",
                resource_type=YdoDispatchLightPartnerConfig,
                required=True,
            )
            polygon_restrictions_json = sdk2.parameters.Resource(
                'polygon restrictions json',
                resource_type=YdoDispatchPolygonRestrictionsJson,
                required=True,
            )

        with sdk2.parameters.Group("Environment") as env_block:
            with sdk2.parameters.RadioGroup('Ydo ctype name') as ctype:
                ctype.values['hamster'] = ctype.Value('hamster')
                ctype.values['testing'] = ctype.Value('testing')
                ctype.values['production'] = ctype.Value('production')

            tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 5 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def monitor(self):
        env = os.environ.copy()
        env["TVM_SECRET"] = self.Parameters.tvm_secret.data()["TVM_SECRET"]

        tmp_dir = tempfile.mkdtemp()
        os.symlink(str(sdk2.ResourceData(self.Parameters.light_partner_config).path), os.path.join(tmp_dir, "light_partner_config.json"))
        os.symlink(
            str(sdk2.ResourceData(self.Parameters.polygon_restrictions_json).path),
            os.path.join(tmp_dir, "polygon_restrictions.json"),
        )
        logging.info("Monitoring...")
        cmd = [
            str(sdk2.ResourceData(self.Parameters.redispatcher_monitoring).path),
            "--config", os.path.join(str(sdk2.ResourceData(self.Parameters.redispatcher_config).path), str(self.Parameters.ctype) + ".yaml"),
        ]
        if self.Parameters.redispatch_date:
            cmd.extend(["--date", self.Parameters.redispatch_date])
        if self.Parameters.trigger_hour:
            cmd.extend(["--trigger_hour", str(self.Parameters.trigger_hour)])
        if self.Parameters.organization_ids:
            cmd.extend(["--organization_ids"] + self.Parameters.organization_ids)
        if self.Parameters.safe_organization_ids:
            cmd.extend(["--safe_organization_ids"] + self.Parameters.safe_organization_ids)

        execute_cmd(
            cmd,
            "ydo_redispatcher_monitoring",
            "Restrictions violated",
            env=env,
            cwd=tmp_dir,
        )

    def on_execute(self):
        with self.memoize_stage.monitor:
            self.monitor()

        logging.info("Done")
