# -*- coding: utf-8 -*-
import logging
import os

import sandbox.common.types.task as ctt
from sandbox import sdk2

from sandbox.projects.ydo import (
    ydo_releasers,
    get_now_utc,
    execute_cmd,
)

from sandbox.projects.geosearch.CleanupYtFolder import clean_folder
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables

from sandbox.projects.ydo.resource_types import (
    YdoSendToSaasLbExecutable,
)

from sandbox.sandboxsdk import environments


class YdoDispatchIndexerExecutable(sdk2.Resource):
    """
        Generate table with proto documents for SAAS
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoDispatchIndexer(sdk2.Task):
    """Task that creates SAAS index and uploads it for service ydo_dispatch_data"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400

        with sdk2.parameters.Group("SAAS specific parameters") as saas_block:
            partner_saas_namespace = sdk2.parameters.String("partner saas namespace (kps)", required=True, default=1)
            customer_saas_namespace = sdk2.parameters.String("customer saas namespace (kps)", required=True, default=2)
            orders_saas_namespace = sdk2.parameters.String("orders saas namespace (kps)", required=True, default=3)

            self_tvm_id = sdk2.parameters.Integer("Self tvm id", required=True)
            tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)

            send_to_saas_lb = sdk2.parameters.Resource(
                "send to saas lb executable",
                resource_type=YdoSendToSaasLbExecutable,
                required=True
            )

            kv_service = sdk2.parameters.String("Service name for kv saas lb", required=True, default="ydo_dispatch_data")
            ctype_saas_lb = sdk2.parameters.String("Ctype saas lb", required=True, default="prestable_ydo")
            logbroker_host = sdk2.parameters.String("Logbroker host", required=True, default="logbroker.yandex.net")
            logbroker_tvm_id = sdk2.parameters.Integer("Destination logbroker tvm id", required=True)
            document_ttl_in_days = sdk2.parameters.Integer("After how many days documents will be deleted", required=True)

        with sdk2.parameters.Group("Required resources") as required_resources_block:
            indexer = sdk2.parameters.Resource(
                "dispatch indexer executable",
                resource_type=YdoDispatchIndexerExecutable,
                required=True
            )

        with sdk2.parameters.Group("Input tables") as required_resources_block:
            partner_rating_table = sdk2.parameters.String("Path to table with patner rating", required=False)
            partner_employee_rating_table = sdk2.parameters.String("Path to table with patner employee rating", required=False)
            partner_boost_table = sdk2.parameters.String("Path to table with patner boost", required=False)
            partner_employee_boost_table = sdk2.parameters.String("Path to table with patner employee boost", required=False)
            order_partner_table = sdk2.parameters.String("Path to table with partner configs", required=False)
            aggregated_orders_table = sdk2.parameters.String("Path to table with aggregated info about orders", required=False)
            external_workers_table = sdk2.parameters.String("Path to table with external workers", required=False)
            orders_prediction_table = sdk2.parameters.String("Path to table with orders prediction", required=False)

            customer_orders_table = sdk2.parameters.String("Path to table with customer orders", required=False)

        with sdk2.parameters.Group("Output parameters") as output_block:
            res_dir = sdk2.parameters.String("Index resulting directory", required=True)

        with sdk2.parameters.Group("Miscellaneous parameters") as miscellaneous_block:
            yt_log_level = sdk2.parameters.String("Yt log level", required=False)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment("yandex-yt"),
        ]
        cores = 1
        disk_space = 5 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def create_index(self):
        yt_token = sdk2.Vault.data(self.owner, "yt-token")

        env = os.environ.copy()
        env["YT_TOKEN"] = yt_token
        env["YT_PROXY"] = "{}.yt.yandex.net".format(self.yt_host)
        env["YT_LOG_LEVEL"] = self.Parameters.yt_log_level if self.Parameters.yt_log_level else "INFO"

        self.Context.index_ts = get_now_utc()

        def table_name(template, kps):
            return os.path.join(self.Parameters.res_dir, template.format(ts=self.Context.index_ts, kps=kps))

        self.Context.out_partner_table = table_name("{ts}_partner_kps={kps}", kps=self.Parameters.partner_saas_namespace)
        self.Context.partner_res_local_path = os.path.join(str(sdk2.Path.cwd()), "partner_id")

        self.Context.out_customer_table = table_name("{ts}_customer_kps={kps}", kps=self.Parameters.customer_saas_namespace)
        self.Context.customer_res_local_path = os.path.join(str(sdk2.Path.cwd()), "customer_id")

        self.Context.out_orders_path = table_name("{ts}_orders_kps={kps}", kps=self.Parameters.orders_saas_namespace)
        self.Context.orders_res_local_path = os.path.join(str(sdk2.Path.cwd()), "orders_predict")

        logging.info("Indexing...")

        cmd = [
            str(sdk2.ResourceData(self.Parameters.indexer).path),
            "--write_proxy", "{}.yt.yandex.net".format(self.yt_host),
            "--partner_res_table", self.Context.out_partner_table,
            "--partner_res_local_path", self.Context.partner_res_local_path,
            "--customer_res_table", self.Context.out_customer_table,
            "--customer_res_local_path", self.Context.customer_res_local_path,
            "--orders_res_table", self.Context.out_orders_path,
            "--orders_res_local_path", self.Context.orders_res_local_path,
        ]

        tables = (
            "partner_rating_table",
            "partner_employee_rating_table",
            "partner_boost_table",
            "partner_employee_boost_table",
            "customer_orders_table",
            "order_partner_table",
            "aggregated_orders_table",
            "external_workers_table",
            "orders_prediction_table",
        )

        for table_name in tables:
            table_path = getattr(self.Parameters, table_name)
            if table_path:
                cmd += ["--{}".format(table_name), table_path]

        execute_cmd(
            cmd,
            "ydo_dispatch_indexer",
            "Failed to create index",
            env=env
        )

    def link(self):
        tables_to_link = {
            self.Context.out_partner_table: os.path.join(self.Parameters.res_dir, "current_partner_kv"),
            self.Context.out_customer_table: os.path.join(self.Parameters.res_dir, "current_customer_kv"),
        }

        self.Context.num_tables = len(tables_to_link)

        link_task = YdoBackupLinkTables(
            self,
            description="Link tables for task {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=self.yt_host,
            yt_vault_token="yt-token",
            yt_tables=tables_to_link
        )
        link_task.enqueue()

        raise sdk2.WaitTask([link_task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True)

    def send_with_saas_lb(self, kps, kv_docs_path, err_key):
        yt_token = sdk2.Vault.data(self.owner, "yt-token")

        env = os.environ.copy()
        env["YT_TOKEN"] = yt_token
        env["TVM_SECRET"] = self.Parameters.tvm_secret.data()["TVM_SECRET"]

        cmd = [
            str(sdk2.ResourceData(self.Parameters.send_to_saas_lb).path),
            "--kv_service", self.Parameters.kv_service,
            "--ctype_saas_lb", self.Parameters.ctype_saas_lb,
            "--logbroker_host", self.Parameters.logbroker_host,
            "--self_tvm_id", str(self.Parameters.self_tvm_id),
            "--logbroker_tvm_id", str(self.Parameters.logbroker_tvm_id),
            "--kps", kps,
            "--read_proxy", "{}.yt.yandex.net".format(self.yt_host),
            "--kv_docs_path", kv_docs_path,
            "--timestamp", str(self.Context.index_ts),
            "--document_ttl_in_days", str(self.Parameters.document_ttl_in_days),
        ]

        execute_cmd(cmd, "ydo_send_to_saas_lb", "Failed to update {} index".format(err_key), env=env)

    def on_execute(self):
        self.yt_host = "hahn"

        with self.memoize_stage.firststage:
            self.create_index()

        with self.memoize_stage.sending_to_lb_partner_snippets:
            self.send_with_saas_lb(kps=self.Parameters.partner_saas_namespace, kv_docs_path=self.Context.partner_res_local_path, err_key="partner")

        with self.memoize_stage.sending_to_lb_customer_snippets:
            self.send_with_saas_lb(kps=self.Parameters.customer_saas_namespace, kv_docs_path=self.Context.customer_res_local_path, err_key="customer")

        with self.memoize_stage.sending_to_lb_orders_snippets:
            self.send_with_saas_lb(kps=self.Parameters.orders_saas_namespace, kv_docs_path=self.Context.orders_res_local_path, err_key="orders")

        with self.memoize_stage.link_to_current:
            self.link()

        with self.memoize_stage.clean_folder:
            clean_folder(self, self.Parameters.res_dir, yt_host=self.yt_host, history_size=4 * self.Context.num_tables)

        logging.info("Done")
