import logging
import os

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.projects.common.geosearch.utils import unpack_file, pack_dir
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo import ydo_releasers, get_now_utc
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables

from sandbox.sandboxsdk import environments


class YdoDocDocBuildRubricatorExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoDocDocBuildWorkersExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoDocDocCache(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoDocDocBuildExport(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]
        dns = ctm.DnsType.DNS64
        cores = 1
        client_tags = ctc.Tag.IVA

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        build_rubricator_executable = sdk2.parameters.Resource(
            'docdoc_build_rubricator',
            resource_type=YdoDocDocBuildRubricatorExecutable,
            required=True,
        )
        build_workers_executable = sdk2.parameters.Resource(
            'docdoc_build_workers',
            resource_type=YdoDocDocBuildWorkersExecutable,
            required=True,
        )
        docdoc_cache = sdk2.parameters.Resource(
            'DocDoc cache',
            resource_type=YdoDocDocCache,
            required=False
        )
        geobase_snapshot = sdk2.parameters.Resource(
            'Geobase 6 snapshot',
            resource_type=GEODATA6BIN_STABLE,
            required=True
        )

        fetch_clinics = sdk2.parameters.Bool('Export clinics` data', default=False)

        res_dir = sdk2.parameters.String('Where to store export', required=True)

        upload_to_stable = sdk2.parameters.Bool('Upload to stable', default=False)

    def export_docdoc_tables(self, yt):
        env = os.environ.copy()
        env['YT_TOKEN'] = yt.config['token']
        env['YT_PROXY'] = 'hahn.yt.yandex.net'
        env['YT_LOG_LEVEL'] = 'DEBUG'

        yt.create('map_node', self.Context.export_dir)

        cache_dir = os.path.join(str(sdk2.Path.cwd()), 'cache')
        cache_dir_archive = cache_dir + '.tar.gz'

        os.mkdir(cache_dir)
        if self.Parameters.docdoc_cache:
            unpack_file(str(sdk2.ResourceData(self.Parameters.docdoc_cache).path), cache_dir)

        docdoc_cache_current = sdk2.ResourceData(YdoDocDocCache(self, 'Ydo DocDoc api requests cache', cache_dir_archive))

        logging.info('Exporting rubrics')
        execute_cmd(
            [
                str(sdk2.ResourceData(self.Parameters.build_rubricator_executable).path),
                '--out_dir', self.Context.export_dir,
                '--write_to_yt',
                '--cache_dir', cache_dir,
            ],
            'docdoc_build_rubricator',
            'Failed to export DocDoc rubrics',
            env=env
        )

        photos_table = os.path.join(self.Context.export_dir, 'photos')
        current_photos_table = os.path.join(self.Context.current_export_dir, 'photos')

        if yt.exists(current_photos_table):
            yt.copy(current_photos_table, photos_table)
        else:
            yt.create('table', photos_table)

        logging.info('Exporting workers')
        execute_cmd(
            [
                str(sdk2.ResourceData(self.Parameters.build_workers_executable).path),
                '--out_dir', self.Context.export_dir,
                '--write_to_yt',
                '--geobase_file', str(sdk2.ResourceData(self.Parameters.geobase_snapshot).path),
                '--debug',
                '--photos_table', photos_table,
                '--cache_dir', cache_dir,
            ] + (['--testing_avatars'] if not self.Parameters.upload_to_stable else [])
              + (['--fetch_clinics'] if self.Parameters.fetch_clinics else []),
            'docdoc_build_workers',
            'Failed to export DocDoc workers',
            env=env
        )
        pack_dir(cache_dir, cache_dir_archive, "tar.gz")
        docdoc_cache_current.ready()

    def link_tables(self, yt):
        tables_to_link = {os.path.join(self.Context.export_dir, table): os.path.join(self.Context.current_export_dir, table) for table in yt.list(self.Context.export_dir)}
        link_task = YdoBackupLinkTables(
            self,
            description='Link tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host='hahn',
            yt_vault_token='yt-token',
            yt_tables=tables_to_link
        )
        link_task.enqueue()

        raise sdk2.WaitTask([link_task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True)

    def clean_folder(self, yt, history_size=14):
        dir_contents = yt.list(self.Parameters.res_dir, absolute=True)
        if len(dir_contents) > history_size:
            for path in sorted(filter(lambda path: not path.endswith('current'), dir_contents), reverse=True)[history_size:]:
                logging.info('Removing path {}'.format(path))
                yt.remove(path, recursive=True)

    def on_execute(self):
        import yt.wrapper as yt

        yt.config['token'] = sdk2.Vault.data(self.owner, 'yt-token')
        yt.config['proxy']['url'] = 'hahn.yt.yandex.net'

        self.Context.export_dir = os.path.join(self.Parameters.res_dir, str(get_now_utc()))
        self.Context.current_export_dir = os.path.join(self.Parameters.res_dir, 'current')

        with self.memoize_stage.export_docdoc:
            self.export_docdoc_tables(yt)

        with self.memoize_stage.link_tables:
            self.link_tables(yt)

        with self.memoize_stage.clean_folder:
            self.clean_folder(yt)
