# -*- coding: utf-8 -*-

import logging
import json

from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.projects.ydo.parameters_view import generate_view


class YdoDryCleaningMailingListProcessor(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/erp/tools/mailing/dry_cleaning_mailing_list/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="dry_cleaning_mailing_list")


class YdoDryCleaningMailingList(sdk2.Task):
    """Task sends daily mailing lists for manufactories"""

    class Parameters(
        generate_view(
            processor=YdoDryCleaningMailingListProcessor,
            check_owner=True,
        )
    ):
        auth_tokens = sdk2.parameters.YavSecret("YAV YDO_OAUTH_TOKENS secret id", required=True)
        mail_tokens = sdk2.parameters.YavSecret("YAV YDO_MAIL_TOKENS secret id", required=True)

        cluster = sdk2.parameters.String('YDB cluster', required=True)
        org_id = sdk2.parameters.String('Organization id', required=True)
        account_slug = sdk2.parameters.String('Sender account slug', default='ya.uslugi', required=True)
        campaign_slug = sdk2.parameters.String('Sender campaign slug', required=True)
        dry_cleaning_org_id_to_mailing_list = sdk2.parameters.Dict(
            'Org id to mailing list', description='Comma separated emails', required=True
        )
        date = sdk2.parameters.String('Report date', required=False, description='format 31/12/2022')

    def on_execute(self):
        logging.info('Start dry cleaning mailing list processor')

        cmd = [
            str(self.Parameters.processor_path),
            '--cluster',
            str(self.Parameters.cluster),
            '--org_id',
            str(self.Parameters.org_id),
            '--account_slug',
            str(self.Parameters.account_slug),
            '--campaign_slug',
            str(self.Parameters.campaign_slug),
            '--yql_token',
            str(self.Parameters.auth_tokens.data()['YQL_TOKEN']),
            '--mail_token',
            str(self.Parameters.mail_tokens.data()['MAIL_TOKEN']),
            '--mailing_mapping',
            json.dumps(self.Parameters.dry_cleaning_org_id_to_mailing_list),
            '--date',
            str(self.Parameters.date),
        ]

        execute_cmd(
            cmd=cmd,
            log_name='dry_cleaning_mailing_list',
            error_message='Failed to create dry cleaning mailing list task',
        )

        logging.info('Done')
