import logging
import os
import tarfile
import tempfile

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE
from sandbox.projects.ydo.resource_types import YdoERPBackendRegionPresets
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.parameters_view import generate_view, ResourceWrapper
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoErpOrdersStatisticsCalculator(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/erp/tools/orders/calculate_orders_statistics/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="calculate_orders_statistics")


class YdoErpCalculateOrdersStatistics(sdk2.Task, SolomonMixinV2):
    """Task that recalculates ERP orders statistics"""

    class Parameters(generate_view(
        processor=YdoErpOrdersStatisticsCalculator,
        geobase_snapshot=ResourceWrapper(GEODATA6BIN_STABLE),
        check_owner=True,
    )):
        tokens = sdk2.parameters.YavSecret("YAV YDO_OAUTH_TOKENS secret id", required=True)

        yt_cluster = sdk2.parameters.String('YT cluster', required=True, default='hahn')
        orders_table_yt_path = sdk2.parameters.String('Path for table orders in YT', required=True)

        ydo_intapi_host = sdk2.parameters.String('Ydo Intapi Host', required=False)
        tvm_id = sdk2.parameters.Integer('Tvm id', required=False)
        tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)
        fail_fast = sdk2.parameters.Bool('Fail fast', default=False)

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting executing orders statistics calculator')

        extracted_tzdata_dir = tempfile.mkdtemp()
        tarfile.open(self.Parameters.geodata_tz_snapshot_path).extractall(extracted_tzdata_dir)

        env = os.environ.copy()
        env['YT_TOKEN'] = self.Parameters.tokens.data()['YT_TOKEN']
        env['YT_PROXY'] = str(self.Parameters.yt_cluster)
        env['TVM_SECRET'] = self.Parameters.tvm_secret.data()['TVM_SECRET']

        execute_cmd(
            [
                str(self.Parameters.processor_path),
                '--ydo_intapi_host', str(self.Parameters.ydo_intapi_host),
                '--tvm_id', str(self.Parameters.tvm_id),
                '--geobase_file', self.Parameters.geobase_snapshot_path,
                '--orders_table_yt_path', str(self.Parameters.orders_table_yt_path),
            ] + (
                ['--fail_fast'] if self.Parameters.fail_fast else []
            ),
            'calculate_orders_statistics',
            'Failed to calculate orders statistics',
            env=env,
        )

        logging.info('Done')

    def on_finish(self, prev_status, status):
        SolomonMixinV2.on_finish(self, prev_status, status)
        super(YdoErpCalculateOrdersStatistics, self).on_finish(prev_status, status)
