import logging
import os

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.parameters_view import generate_view, ResourceWrapper
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2
from sandbox.projects.ydo.erp import YdoErpEducationCoursesData
from sandbox.projects.ydo.erp.YdoErpServicesSchemasMerger import YdoErpServicesSchemasJson


class YdoErpMigrateServiceCardsExecutable(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/erp/tools/services/migrate_service_cards/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="migrate_service_cards")


class YdoErpMigrateServiceCards(sdk2.Task, SolomonMixinV2):
    """Task that migrates ERP organizations and employees service cards on rubricator changes"""

    class Parameters(generate_view(
        processor=YdoErpMigrateServiceCardsExecutable,
        services_schemas=ResourceWrapper(YdoErpServicesSchemasJson),
        education_courses=ResourceWrapper(YdoErpEducationCoursesData),
        check_owner=True,
    )):
        ydb_host = sdk2.parameters.String('YDB Host', required=True)

        ydb_port = sdk2.parameters.Integer('YDB Port', default=2135, required=True)

        ydb_database = sdk2.parameters.String('YDB Database', required=True)

        ydb_home = sdk2.parameters.String('YDB Home', required=True)

        ydb_token = sdk2.parameters.YavSecret("YDB token in yav", required=True)

        logbroker_token = sdk2.parameters.YavSecret("Logbroker token in yav", required=True)

        education_courses_env = sdk2.parameters.RadioGroup(
            'Education courses environment',
            choices=(
                ('production', 'production'),
                ('testing', 'testing'),
            ),
            required=True,
            default='testing',
        )

        replication_async_api_env = sdk2.parameters.RadioGroup(
            'Replication Async Api Environment',
            choices=(
                ('production', 'production'),
                ('testing', 'testing'),
            ),
            required=True,
            default='testing',
        )

        organizations_blacklist = sdk2.parameters.List('Organization IDs to skip')

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting executing service cards migration')

        env = os.environ.copy()
        env['YDB_TOKEN'] = self.Parameters.ydb_token.data()['YDB_TOKEN']
        env['LOGBROKER_TOKEN'] = self.Parameters.ydb_token.data()['LOGBROKER_TOKEN']

        education_courses_path = os.path.join(self.Parameters.education_courses_path, str(self.Parameters.education_courses_env), 'education_courses.json')

        cmd = [
            str(self.Parameters.processor_path),
            '--ydb_host', str(self.Parameters.ydb_host),
            '--ydb_port', str(self.Parameters.ydb_port),
            '--ydb_database', str(self.Parameters.ydb_database),
            '--ydb_home', str(self.Parameters.ydb_home),
            '--services_schemas', str(self.Parameters.services_schemas_path),
            '--education_courses', education_courses_path,
            '--replication-async-api-pipe-env', str(self.Parameters.replication_async_api_env),
        ]

        if self.Parameters.organizations_blacklist:
            cmd.extend(['--organizations_blacklist', ' '.join(self.Parameters.organizations_blacklist)])

        execute_cmd(
            cmd,
            'migrate_service_cards',
            'Failed to migrate service cards',
            env=env,
        )

        logging.info('Done')

    def on_finish(self, prev_status, status):
        SolomonMixinV2.on_finish(self, prev_status, status)
        super(YdoErpMigrateServiceCards, self).on_finish(prev_status, status)
