import logging
import os

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoErpScheduleConfirmationProcessor(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/erp/tools/employees/schedule_confirmation_processor/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="schedule_confirmation_processor")


class YdoErpProcessScheduleConfirmations(sdk2.Task, SolomonMixinV2):
    """Task that processes ERP employees schedule confirmations and sends notifications"""

    class Parameters(generate_view(
        processor=YdoErpScheduleConfirmationProcessor,
        check_owner=True,
    )):
        ydo_intapi_host = sdk2.parameters.String('Ydo IntApi Host', required=True)

        tvm_id = sdk2.parameters.Integer('Ydo Tvm Id', required=True)

        tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)

        employees_batch_size = sdk2.parameters.Integer('Employees batch size [1; 1000]', default=1000, required=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting executing schedule confirmation processor')

        env = os.environ.copy()
        env['TVM_SECRET'] = self.Parameters.tvm_secret.data()['TVM_SECRET']

        execute_cmd(
            [
                str(self.Parameters.processor_path),
                '--ydo-intapi-host', str(self.Parameters.ydo_intapi_host),
                '--tvm-id', str(self.Parameters.tvm_id),
                '--employees-batch-size', str(self.Parameters.employees_batch_size),
            ],
            'process_schedule_confirmations',
            'Failed to process schedule confirmations',
            env=env,
        )

        logging.info('Done')

    def on_finish(self, prev_status, status):
        SolomonMixinV2.on_finish(self, prev_status, status)
        super(YdoErpProcessScheduleConfirmations, self).on_finish(prev_status, status)
