import logging
import os

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoErpTrackMarkupOfScheduleInfoRequestIntervalProcessor(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/erp/tools/employees/track_markup_of_schedule_info_request_interval_processor/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="track_markup_of_schedule_info_request_interval_processor")


class YdoErpProcessTrackMarkupOfScheduleInfoRequestInterval(sdk2.Task, SolomonMixinV2):
    """Task that processes ERP employees schedule confirmations and sends notifications"""

    class Parameters(generate_view(
        processor=YdoErpTrackMarkupOfScheduleInfoRequestIntervalProcessor,
        check_owner=True,
    )):
        ydb_host = sdk2.parameters.String('YDB Host', required=True)

        ydb_port = sdk2.parameters.Integer('YDB Port', default=2135, required=True)

        ydb_database = sdk2.parameters.String('YDB Database', required=True)

        ydb_home = sdk2.parameters.String('YDB Home', required=True)

        ydb_token = sdk2.parameters.YavSecret("YDB token in yav", required=True)

        async_api_env = sdk2.parameters.RadioGroup(
            'Async Api Environment',
            choices=(
                ('production', 'production'),
                ('testing', 'testing'),
            ),
            required=True,
            default='testing',
        )

        logbroker_token = sdk2.parameters.YavSecret("Logbroker token in yav", required=True)

        replication_async_api_env = sdk2.parameters.RadioGroup(
            'Replication Async Api Environment',
            choices=(
                ('production', 'production'),
                ('testing', 'testing'),
            ),
            required=True,
            default='testing',
        )

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting executing track markup of schedule info request interval processor')

        env = os.environ.copy()
        env['YDB_TOKEN'] = self.Parameters.ydb_token.data()['YDB_TOKEN']
        env['LOGBROKER_TOKEN'] = self.Parameters.ydb_token.data()['LOGBROKER_TOKEN']

        execute_cmd(
            [
                str(self.Parameters.processor_path),
                '--ydb-host', str(self.Parameters.ydb_host),
                '--ydb-port', str(self.Parameters.ydb_port),
                '--ydb-database', str(self.Parameters.ydb_database),
                '--ydb-home', str(self.Parameters.ydb_home),
                '--async-api-env', str(self.Parameters.async_api_env),
                '--replication-async-api-pipe-env', str(self.Parameters.replication_async_api_env),
            ],
            'process_track_markup_of_schedule_info_request_interval',
            'Failed to process track markup of schedule info request interval',
            env=env,
        )

        logging.info('Done')

    def on_finish(self, prev_status, status):
        SolomonMixinV2.on_finish(self, prev_status, status)
        super(YdoErpProcessTrackMarkupOfScheduleInfoRequestInterval, self).on_finish(prev_status, status)
